package ru.yandex.direct.core.entity.banner.service.validation;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.banner.model.ImageType;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.banner.model.ImageType.REGULAR;
import static ru.yandex.direct.core.entity.banner.model.ImageType.SMALL;
import static ru.yandex.direct.core.entity.banner.model.ImageType.WIDE;
import static ru.yandex.direct.core.entity.banner.type.bannerimage.BannerWithBannerImageConstraints.bannerImageExists;
import static ru.yandex.direct.core.entity.banner.type.bannerimage.BannerWithBannerImageConstraints.isBannerImageTypeValid;

@ParametersAreNonnullByDefault
public class BannerImageHashValidator implements Validator<String, Defect> {

    private final Map<String, ImageType> typedHashes;
    private final Set<ImageType> allowedImageTypes;

    /**
     * @param typedHashes - мапа хэш изображения -> тип, данные получены из бд
     */
    public BannerImageHashValidator(Map<String, ImageType> typedHashes,
                                       boolean regularImageAllowed,
                                       boolean smallImageAllowed) {
        this(typedHashes, getAllowedImageTypes(regularImageAllowed, smallImageAllowed));
    }

    public BannerImageHashValidator(Map<String, ImageType> typedHashes,
                                       Set<ImageType> allowedImageTypes) {
        this.typedHashes = typedHashes;
        this.allowedImageTypes = allowedImageTypes;
    }

    @Override
    public ValidationResult<String, Defect> apply(@Nullable String imageHash) {
        ItemValidationBuilder<String, Defect> vb = ModelItemValidationBuilder.of(imageHash);
        if (imageHash != null) {

            vb.check(bannerImageExists(typedHashes.keySet()));

            ImageType imageType = typedHashes.get(imageHash);
            vb.check(isBannerImageTypeValid(imageType, allowedImageTypes), When.isValid());
        }
        return vb.getResult();
    }

    private static Set<ImageType> getAllowedImageTypes(boolean regularImageAllowed, boolean smallImageAllowed) {
        Set<ImageType> allowedImageTypes = new HashSet<>();
        allowedImageTypes.add(WIDE);
        if (smallImageAllowed) {
            allowedImageTypes.add(SMALL);
        }
        if (regularImageAllowed) {
            allowedImageTypes.add(REGULAR);
        }

        return allowedImageTypes;
    }
}
