package ru.yandex.direct.core.entity.banner.service.validation;

import java.util.Arrays;
import java.util.regex.Pattern;

import com.google.common.base.CharMatcher;

import ru.yandex.direct.utils.TextConstants;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static com.google.common.base.CharMatcher.anyOf;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static ru.yandex.direct.core.entity.banner.service.BannerUtils.templateCountIn;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerLettersConstants.ALLOW_BANNER_LETTERS_STR;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstants.MAX_NUMBER_OF_NARROW_CHARACTERS;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstants.NARROW_SYMBOLS;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstants.TEMPLATE_MARKER_SYMBOL;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.cannotHaveTemplate;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.illegalComma;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.invalidTemplateQuery;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.maxLengthWordTemplateMarker;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.maxNumberOfNarrowCharacters;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.maxTextLengthWithoutTemplateMarker;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.restrictedCharsInField;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.stringShouldNotBeBlank;
import static ru.yandex.direct.utils.TextConstants.SPACE_CHARS;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

public class BannerTextConstraints {

    private static final CharMatcher ALLOW_BANNER_LETTERS_MATCHER = anyOf(ALLOW_BANNER_LETTERS_STR);

    private BannerTextConstraints() {
        // only static methods
    }

    /**
     * Проверка того, что строка содержит хотя бы один непробельный символ
     */
    public static Constraint<String, Defect> stringIsNotBlank() {
        return notBlank().overrideDefect(stringShouldNotBeBlank());
    }

    /**
     * Проверка допустимых символов
     */
    public static Constraint<String, Defect> charsAreAllowed() {
        return fromPredicate(ALLOW_BANNER_LETTERS_MATCHER::matchesAllOf, restrictedCharsInField());
    }

    /**
     * Проверка валидности шаблона
     */
    public static Constraint<String, Defect> templateIsValid() {
        return fromPredicate(x -> templateCountIn(x) <= 1, invalidTemplateQuery());
    }

    /**
     * Проверка отсутствия шаблонов
     */
    public static Constraint<String, Defect> withoutTemplates() {
        return fromPredicate(x -> templateCountIn(x) == 0, cannotHaveTemplate());
    }

    /**
     * Проверка максимальной длины без учёта вхождения шаблонных меток
     * {@link BannerTextConstants#TEMPLATE_MARKER_SYMBOL} (самостоятельный диез учитывается) и узких символов
     */
    public static Constraint<String, Defect> hasValidLengthWithoutTemplateMarkerAndNarrowCharacters(
            int maxLength) {
        return fromPredicate(x -> x.replaceAll(String.format("[%s]", NARROW_SYMBOLS), EMPTY).length()
                        <= maxLength + 2 * templateCountIn(x),
                maxTextLengthWithoutTemplateMarker(maxLength));
    }

    /**
     * Проверка максимальной длины без учёта вхождения шаблонных меток
     * {@link BannerTextConstants#TEMPLATE_MARKER_SYMBOL} (самостоятельный диез учитывается)
     */
    public static Constraint<String, Defect> hasValidLengthWithoutTemplateMarker(int maxLength) {
        return fromPredicate(x -> x.length() <= maxLength + 2 * templateCountIn(x),
                maxTextLengthWithoutTemplateMarker(maxLength));
    }

    /**
     * Проверка количества узких символов
     */
    public static Constraint<String, Defect> numberOfNarrowCharactersIsValid() {
        return numberOfNarrowCharactersIsValid(MAX_NUMBER_OF_NARROW_CHARACTERS);
    }

    /**
     * Проверка количества узких символов
     */
    public static Constraint<String, Defect> numberOfNarrowCharactersIsValid(int max) {
        return fromPredicate(x -> x.replaceAll(String.format("[^%s]", NARROW_SYMBOLS), "").length() <= max,
                maxNumberOfNarrowCharacters(max));
    }

    /**
     * Проверка максимальной длины слова без учёта вхождения шаблонных меток
     * {@link BannerTextConstants#TEMPLATE_MARKER_SYMBOL} (самостоятельный диез учитывается)
     */
    public static Constraint<String, Defect> wordsHaveValidLength(int maxLengthWord) {
        return fromPredicate(x -> Arrays.stream(x.split("[" + TEMPLATE_MARKER_SYMBOL + TextConstants.SPACE_CHARS +
                        "\\-]"))
                        .noneMatch(w -> w.length() > maxLengthWord),
                maxLengthWordTemplateMarker(maxLengthWord));
    }

    /**
     * Проверка незакрепленной запятой: (пробел, неразрывный пробел) перед запятой
     */
    public static Constraint<String, Defect> commasAreBounded() {
        return fromPredicate(
                x -> !Pattern.compile(String.format("[%1$s],[^%1$s]", SPACE_CHARS)).matcher(x).find(),
                illegalComma());
    }
}
