package ru.yandex.direct.core.entity.banner.service.validation;

import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstraints.charsAreAllowed;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstraints.commasAreBounded;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstraints.hasValidLengthWithoutTemplateMarker;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstraints.hasValidLengthWithoutTemplateMarkerAndNarrowCharacters;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstraints.numberOfNarrowCharactersIsValid;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstraints.stringIsNotBlank;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstraints.templateIsValid;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstraints.withoutTemplates;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstraints.wordsHaveValidLength;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

public class BannerTextValidator implements Validator<String, Defect> {

    /**
     * Максимальная длина.
     * <p>
     * Если проверка макс. количества "узких" символов выключена,
     * то длина проверяется вместе с ними, то есть такие символы не рассматриваются отдельно от остальных.
     * <p>
     * Если проверка макс. количества "узких" символов включена,
     * то длина проверяется без узких символов, плюс отдельно проверяется количество узких символов.
     */
    private final int maxLength;

    /**
     * Максимальная длина слова.
     */
    private final int maxWordLength;

    /**
     * Максимальное количество "узких" символов.
     * Если null, то такие символы не учитываются отдельно,
     * а проверяются на равне с остальными символами при проверке длины.
     *
     * @see #maxLength
     */
    private final int maxNumberOfNarrowCharacters;

    private final boolean checkNarrowChars;

    /**
     * Разрешены ли шаблоны в данном поле.
     */
    private final boolean isTemplateAllowed;

    private final Defect stringIsNotBlankDefect;

    private BannerTextValidator(Builder b) {
        this.maxLength = b.maxLength;
        this.maxWordLength = b.maxWordLength;
        this.maxNumberOfNarrowCharacters = b.maxNumberOfNarrowCharacters;
        this.checkNarrowChars = b.checkNarrowChars;
        this.isTemplateAllowed = b.isTemplateAllowed;
        this.stringIsNotBlankDefect = b.stringIsNotBlankDefect;
    }

    public static Builder builder(int maxLength, int maxWordLength) {
        return new Builder(maxLength, maxWordLength);
    }

    @Override
    public ValidationResult<String, Defect> apply(String title) {
        ItemValidationBuilder<String, Defect> builder = ItemValidationBuilder.of(title);
        builder
                .check(notNull())
                .check(stringIsNotBlank(), stringIsNotBlankDefect)
                .check(charsAreAllowed())
                .check(wordsHaveValidLength(maxWordLength))
                .check(commasAreBounded());

        if (isTemplateAllowed) {
            builder.check(templateIsValid());
        } else {
            builder.check(withoutTemplates());
        }

        if (checkNarrowChars) {
            builder
                    .check(hasValidLengthWithoutTemplateMarkerAndNarrowCharacters(maxLength))
                    .check(numberOfNarrowCharactersIsValid(maxNumberOfNarrowCharacters));
        } else {
            builder.check(hasValidLengthWithoutTemplateMarker(maxLength));
        }

        return builder.getResult();
    }

    public static final class Builder {
        /**
         * Максимальная длина.
         * <p>
         * Если проверка макс. количества "узких" символов выключена,
         * то длина проверяется вместе с ними, то есть такие символы не рассматриваются отдельно от остальных.
         * <p>
         * Если проверка макс. количества "узких" символов включена,
         * то длина проверяется без узких символов, плюс отдельно проверяется количество узких символов.
         */
        private final int maxLength;

        /**
         * Максимальная длина слова.
         */
        private final int maxWordLength;

        /**
         * Максимальное количество "узких" символов.
         * Если null, то такие символы не учитываются отдельно,
         * а проверяются на равне с остальными символами при проверке длины.
         *
         * @see #maxLength
         */
        private int maxNumberOfNarrowCharacters;
        public boolean checkNarrowChars = false;
        public boolean isTemplateAllowed = true;
        private Defect stringIsNotBlankDefect = null;

        private Builder(int maxLength, int maxWordLength) {
            this.maxLength = maxLength;
            this.maxWordLength = maxWordLength;
        }

        public Builder withMaxNumberOfNarrowCharacters(int maxNumberOfNarrowCharacters) {
            this.maxNumberOfNarrowCharacters = maxNumberOfNarrowCharacters;
            this.checkNarrowChars = true;
            return this;
        }

        public Builder withTemplateDisallowed() {
            isTemplateAllowed = false;
            return this;
        }

        /**
         * @param stringIsNotBlankDefect тип дефекта для пустой строки
         */
        public Builder withStringIsNotBlankDefect(Defect stringIsNotBlankDefect) {
            this.stringIsNotBlankDefect = stringIsNotBlankDefect;
            return this;
        }

        public BannerTextValidator build() {
            return new BannerTextValidator(this);
        }
    }

}
