package ru.yandex.direct.core.entity.banner.service.validation;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainer;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.AccessDefectPresets;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignAccessDefects;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessConstraint;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;

import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.actionInArchivedCampaign;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.adNotFound;
import static ru.yandex.direct.validation.defect.CollectionDefects.duplicatedObject;

@ParametersAreNonnullByDefault
@Component
class BannersAccessValidator {

    public static final CampaignAccessDefects ACCESS_DEFECTS =
            AccessDefectPresets.AD_ACCESS_DEFECTS.toBuilder()
                    .withArchivedModification(actionInArchivedCampaign())
                    .build();

    private final ModelChangesValidationTool updateValidationTool;
    private final CampaignSubObjectAccessCheckerFactory campaignAccessCheckerFactory;

    @Autowired
    public BannersAccessValidator(
            CampaignSubObjectAccessCheckerFactory campaignAccessCheckerFactory) {
        this.campaignAccessCheckerFactory = campaignAccessCheckerFactory;

        this.updateValidationTool = ModelChangesValidationTool.builder()
                .minSize(1)
                .duplicatedItemDefect(duplicatedObject())
                .objectNotFoundDefect(adNotFound())
                .build();
    }

    public <T extends Banner> void apply(
            BannersUpdateOperationContainer container,
            ValidationResult<List<ModelChanges<T>>, Defect> vr,
            Collection<Long> modelChangesBannerIds,
            Set<Long> clientBannerIds) {
        if (vr.getValue().size() == 0) {
            return;
        }
        updateValidationTool.validateModelChangesList(vr, () -> clientBannerIds);

        var lvb = new ListValidationBuilder<>(vr);
        if (!container.isPartOfComplexOperation()) {
            CampaignSubObjectAccessConstraint accessConstraint = campaignAccessCheckerFactory
                    .newAdsChecker(container.getOperatorUid(), container.getClientId(), modelChangesBannerIds)
                    .createValidator(CampaignAccessType.READ_WRITE, ACCESS_DEFECTS)
                    .getAccessConstraint();
            lvb.checkEachBy(mc -> validateUpdateRights(mc, accessConstraint), When.isValid());
        }
    }

    private <T extends Banner> ValidationResult<ModelChanges<T>, Defect> validateUpdateRights(
            ModelChanges<T> modelChanges,
            CampaignSubObjectAccessConstraint constraint) {
        ItemValidationBuilder<ModelChanges<T>, Defect> vb =
                ItemValidationBuilder.of(modelChanges);

        vb.item(modelChanges.getId(), Banner.ID.name()).check(constraint);

        return vb.getResult();
    }

}
