package ru.yandex.direct.core.entity.banner.service.validation;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainer;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.inconsistentBannerType;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@Service
@ParametersAreNonnullByDefault
public class BannersUpdateValidationService {

    private final BannersAccessValidator accessValidator;

    @Autowired
    public BannersUpdateValidationService(BannersAccessValidator accessValidator) {
        this.accessValidator = accessValidator;
    }

    public <T extends Banner> ValidationResult<List<ModelChanges<T>>, Defect> validateModelChanges(
            BannersUpdateOperationContainer operationContainer,
            List<ModelChanges<T>> modelChanges,
            Collection<Long> modelChangesBannerIds,
            Set<Long> clientBannerIds,
            Function<BannersUpdateOperationContainer, Constraint<ModelChanges<T>, Defect>>
                    modelChangesIsApplicableConstraintCreator) {
        ValidationResult<List<ModelChanges<T>>, Defect> vr = new ValidationResult<>(modelChanges);
        accessValidator.apply(operationContainer, vr, modelChangesBannerIds, clientBannerIds);
        Constraint<ModelChanges<T>, Defect> modelChangesIsApplicable = modelChangesIsApplicableConstraintCreator
                .apply(operationContainer);
        new ListValidationBuilder<>(vr)
                .checkEach(modelChangesHasConsistentType(operationContainer), When.isValid())
                .checkEach(modelChangesIsApplicable, When.isValid());
        return vr;
    }

    private static <T extends Banner> Constraint<ModelChanges<T>, Defect> modelChangesHasConsistentType(
            BannersUpdateOperationContainer container) {
        return fromPredicate(mc -> {
            Class<? extends T> modelChangesClass = mc.getModelType();
            Class<? extends Banner> runtimeClass = container.getRuntimeClass(mc.getId());
            return runtimeClass != null && modelChangesClass.isAssignableFrom(runtimeClass);
        }, inconsistentBannerType());
    }

}
