package ru.yandex.direct.core.entity.banner.service.validation;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerStatusPostModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.bs.export.queue.repository.BsExportQueueRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.AccessDefectPresets;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignAccessDefects;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessConstraint;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.utils.NumberUtils;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.adNotFound;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.unableToDelete;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.defect.CollectionDefects.duplicatedObject;

/**
 * Сервис валидации для операции удаления баннеров.
 */
@Service
public class DeleteBannerValidationService {

    private static final ru.yandex.direct.core.entity.adgroup.model.StatusModerate ADGROUP_MODERATE_YES =
            ru.yandex.direct.core.entity.adgroup.model.StatusModerate.YES;
    private static final ru.yandex.direct.core.entity.adgroup.model.StatusPostModerate ADGROUP_POST_MODERATE_REJECTED =
            ru.yandex.direct.core.entity.adgroup.model.StatusPostModerate.REJECTED;
    private static final ru.yandex.direct.core.entity.adgroup.model.StatusPostModerate ADGROUP_POST_MODERATE_YES =
            ru.yandex.direct.core.entity.adgroup.model.StatusPostModerate.YES;

    private static final Set<ru.yandex.direct.core.entity.adgroup.model.StatusPostModerate>
            ADGROUP_STATUS_POST_MODERATE =
            new HashSet<>(Arrays.asList(ADGROUP_POST_MODERATE_YES, ADGROUP_POST_MODERATE_REJECTED));
    private static final Set<BannerStatusPostModerate> BANNER_STATUSES_POST_MODERATE =
            new HashSet<>(Arrays.asList(BannerStatusPostModerate.YES, BannerStatusPostModerate.REJECTED));
    private static final CampaignAccessDefects ACCESS_DEFECTS = AccessDefectPresets.AD_ACCESS_DEFECTS;

    private final BannerTypedRepository bannerTypedRepository;
    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;
    private final BsExportQueueRepository  bsExportQueueRepository;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;

    @Autowired
    public DeleteBannerValidationService(
            BannerTypedRepository bannerTypedRepository,
            AdGroupRepository adGroupRepository,
            CampaignRepository campaignRepository,
            BsExportQueueRepository bsExportQueueRepository,
            CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory) {
        this.bannerTypedRepository = bannerTypedRepository;
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.bsExportQueueRepository = bsExportQueueRepository;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
    }

    public ValidationResult<List<Long>, Defect> validateDelete(int shard, Long operatorUid, ClientId clientId,
                                                               List<Long> requestedBannerIds) {
        ListValidationBuilder<Long, Defect> lvb = validateAccess(operatorUid, clientId, requestedBannerIds);

        List<Long> validBannerIds = ValidationResult.getValidItems(lvb.getResult());
        List<BannerWithSystemFields> visibleBanners =
                bannerTypedRepository.getStrictlyFullyFilled(shard, validBannerIds, BannerWithSystemFields.class);

        lvb
                .checkEach(inSet(listToSet(visibleBanners, BannerWithSystemFields::getId)), adNotFound(),
                        When.isValid())
                .checkEach(validId(), adNotFound(), When.isValid())
                .checkEach(unique(), duplicatedObject(), When.isValid())
                .checkEach(notInBs(visibleBanners), When.isValid())
                .checkEach(notInCampaignInBsQueue(shard, visibleBanners), When.isValid())
                .checkEach(notWithMoneyAndModerated(shard, clientId, visibleBanners), When.isValid());
        return lvb.getResult();
    }

    /**
     * Проверяет права доступа к баннерам по списку ID и возвращает только доступные клиенту значения.
     *
     * @param operatorUid UID оператора
     * @param clientId    ID клиента
     * @param bannerIds   Список ID баннеров
     * @return Список доступных клиенту ID баннеров
     */
    private ListValidationBuilder<Long, Defect> validateAccess(Long operatorUid, ClientId clientId,
                                                               List<Long> bannerIds) {
        CampaignSubObjectAccessConstraint accessConstraint = campaignSubObjectAccessCheckerFactory
                .newAdsChecker(operatorUid, clientId, bannerIds)
                .createValidator(CampaignAccessType.READ_WRITE, ACCESS_DEFECTS)
                .getAccessConstraint();
        ListValidationBuilder<Long, Defect> lvb = ListValidationBuilder.of(bannerIds);
        lvb
                .check(notNull())
                .checkEach(notNull(), When.isValid())
                .checkEach(accessConstraint, When.isValid());
        return lvb;
    }

    /**
     * Предикат для {@link #notWithMoneyAndModerated}
     * Используется так же для вычисления флага возможности удаления баннера
     *
     * @see #getCanDeleteBannersPredicate(int, ClientId, List)
     */
    private Predicate<Long> notWithMoneyAndModeratedPredicate(int shard, ClientId clientId,
                                                              List<BannerWithSystemFields> banners) {
        List<Long> adGroupIds = mapList(banners, BannerWithSystemFields::getAdGroupId);
        List<AdGroup> adGroups = adGroupRepository.getAdGroups(shard, adGroupIds);
        Map<Long, AdGroup> adGroupById = listToMap(adGroups, AdGroup::getId);

        List<Long> campaignIds = mapList(banners, BannerWithSystemFields::getCampaignId);
        Map<Long, BigDecimal> campaignsSum = campaignRepository.getCampaignsSum(shard, clientId, campaignIds);

        Set<Long> bannersSuccessModerate = StreamEx.of(banners)
                .filter(b -> NumberUtils.greaterThanZero(campaignsSum.get(b.getCampaignId())))
                .filter(b -> isBannerSuccessfullyModerated(b, adGroupById.get(b.getAdGroupId())))
                .map(BannerWithSystemFields::getId)
                .toSet();
        return bannerId -> !bannersSuccessModerate.contains(bannerId);
    }

    /**
     * Проверяет что баннер не находится в кампании с деньгами, или хотя бы не прошел модерацию.
     *
     * @param shard   Шард
     * @param banners Список баннеров
     */
    private Constraint<Long, Defect> notWithMoneyAndModerated(int shard, ClientId clientId,
                                                              List<BannerWithSystemFields> banners) {
        return Constraint.fromPredicate(notWithMoneyAndModeratedPredicate(shard, clientId, banners), unableToDelete());
    }

    /**
     * Проверяет возможность удаления баннеров на основании его статуса модерации.
     */
    private boolean isBannerSuccessfullyModerated(BannerWithSystemFields banner, AdGroup adGroup) {
        if (banner.getStatusModerate() == BannerStatusModerate.YES && adGroup.getStatusModerate() == ADGROUP_MODERATE_YES) {
            return true;
        }

        BannerStatusPostModerate bannerPostModerate = banner.getStatusPostModerate();
        if (BANNER_STATUSES_POST_MODERATE.contains(bannerPostModerate)) {
            return true;
        }
        return ADGROUP_STATUS_POST_MODERATE.contains(adGroup.getStatusPostModerate())
                && bannerPostModerate != BannerStatusPostModerate.NO;
    }

    /**
     * Предикат для {@link #notInCampaignInBsQueue}
     * Используется так же для вычисления флага возможности удаления баннера
     *
     * @see #getCanDeleteBannersPredicate(int, ClientId, List)
     */
    private Predicate<Long> notInCampaignInBsQueuePredicate(int shard, List<BannerWithSystemFields> banners) {
        Set<Long> campaignsInBsQueue = bsExportQueueRepository.getCampaignsIdsInQueue(shard,
                mapList(banners, BannerWithSystemFields::getCampaignId));
        Set<Long> bannersInBsQueue = banners.stream()
                .filter(b -> campaignsInBsQueue.contains(b.getCampaignId())
                        && !(b.getStatusModerate().equals(BannerStatusModerate.NEW)
                        && b.getStatusPostModerate().equals(BannerStatusPostModerate.NO)))
                .map(BannerWithSystemFields::getId).collect(Collectors.toSet());

        return bannerId -> !bannersInBsQueue.contains(bannerId);
    }

    /**
     * Проверяет что баннеры не находятся в кампании в очереди на отправку в БК, или что баннер хотя бы является
     * черновиком
     *
     * @param shard   Шард
     * @param banners Список баннеров
     */
    private Constraint<Long, Defect> notInCampaignInBsQueue(int shard, List<BannerWithSystemFields> banners) {
        return Constraint.fromPredicate(notInCampaignInBsQueuePredicate(shard, banners), unableToDelete());
    }

    /**
     * Предикат для {@link #notInBs}
     * Используется так же для вычисления флага возможности удаления баннера
     *
     * @see #getCanDeleteBannersPredicate(int, ClientId, List)
     */
    private Predicate<Long> notInBsPredicate(List<BannerWithSystemFields> banners) {
        Set<Long> bsBanners = banners.stream()
                .filter(b -> b.getBsBannerId() != 0L)
                .map(BannerWithSystemFields::getId)
                .collect(Collectors.toSet());
        return bannerId -> !bsBanners.contains(bannerId);
    }

    /**
     * Проверяет что баннеры в данный момент не показываются в БК.
     *
     * @param banners Список баннеров
     */
    private Constraint<Long, Defect> notInBs(List<BannerWithSystemFields> banners) {
        return Constraint.fromPredicate(notInBsPredicate(banners), unableToDelete());
    }

    /**
     * Предикат для вычисления возможности удаления баннеров
     */
    public Predicate<Long> getCanDeleteBannersPredicate(int shard, ClientId clientId,
                                                        List<BannerWithSystemFields> banners) {
        return notInBsPredicate(banners)
                .and(notInCampaignInBsQueuePredicate(shard, banners))
                .and(notWithMoneyAndModeratedPredicate(shard, clientId, banners));
    }

}
