package ru.yandex.direct.core.entity.banner.service.validation;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessConstraint;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.moderationreason.service.ModerationReasonService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.moderateArchivedBanner;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.moderateBannerInAdGroupWithoutShowConditions;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.moderateNonDraftBanner;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.remoderateBannerIsNotAllowed;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.remoderateDraftBanner;
import static ru.yandex.direct.feature.FeatureName.CLIENT_ALLOWED_TO_REMODERATE;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class ModerateBannerValidationService {
    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final ModerationReasonService moderationReasonService;
    private final RbacService rbacService;

    public ModerateBannerValidationService(AdGroupRepository adGroupRepository,
                                           CampaignRepository campaignRepository,
                                           CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
                                           ModerationReasonService moderationReasonService, RbacService rbacService) {
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.moderationReasonService = moderationReasonService;
        this.rbacService = rbacService;
    }

    private ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> validate(
            int shard,
            ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> preValidateResult,
            Map<Long, BannerWithSystemFields> models) {
        Set<Long> adGroupsWithConditions = adGroupRepository.getAdGroupIdsWithConditions(shard,
                mapList(models.values(), BannerWithSystemFields::getAdGroupId));
        Set<Long> bannersWithConditions = StreamEx.of(models.values())
                .filter(b -> adGroupsWithConditions.contains(b.getAdGroupId()))
                .map(BannerWithSystemFields::getId)
                .toSet();

        return new ListValidationBuilder<>(preValidateResult)
                .checkEach(bannerNotArchived(models), When.isValid())
                .checkEach(adGroupHasShowConditions(bannersWithConditions), When.isValid())
                .getResult();
    }

    /**
     * Валидация для первичной модерации объявлений.
     */
    public ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> validateModerate(
            int shard, ClientId clientId,
            ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> preValidateResult,
            Map<Long, BannerWithSystemFields> models) {

        preValidateResult = validate(shard, preValidateResult, models);

        Map<Long, Long> archivedCampaignIdByBannerId = EntryStream
                .of(campaignRepository.getCampaignsWithTypeByBannerIds(shard, clientId, models.keySet()))
                .filterValues(CampaignWithType::isArchived)
                .mapValues(CampaignWithType::getId)
                .toMap();

        return new ListValidationBuilder<>(preValidateResult)
                .checkEach(campaignNotArchived(archivedCampaignIdByBannerId), When.isValid())
                .checkEach(bannerIsDraft(models), When.isValid())
                .getResult();
    }

    /**
     * Валидация для повторной модерации объявлений.
     */
    public ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> validateReModerate(
            int shard, ClientId clientId,
            ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> preValidateResult,
            Map<Long, BannerWithSystemFields> models) {

        preValidateResult = validate(shard, preValidateResult, models);
        var campaignsWithTypeByBannerIds = campaignRepository.getCampaignsWithTypeByBannerIds(shard, clientId,
                models.keySet());

        var campaignIds = listToSet(campaignsWithTypeByBannerIds.values(), CampaignWithType::getId);
        var simpleCampaigns = campaignRepository.getCampaignsSimple(shard, campaignIds);

        //Перемодерировать объявления можно в кампаниях, которые не черновики, и не были архивированы.
        Map<Long, CampaignSimple> cannotBeRemoderatedCampaignIdByBannerId = EntryStream
                .of(campaignsWithTypeByBannerIds)
                .mapValues(campaignWithType -> simpleCampaigns.get(campaignWithType.getId()))
                .filterValues(campaignSimple -> campaignSimple.getStatusModerate() == CampaignStatusModerate.NEW
                        || campaignSimple.getStatusArchived())
                .toMap();

        var archivedCampaignIdByBannerId = EntryStream.of(cannotBeRemoderatedCampaignIdByBannerId)
                .filterValues(CampaignSimple::getStatusArchived)
                .mapValues(CampaignSimple::getId)
                .toMap();

        var draftCampaignIdByBannerId = EntryStream.of(cannotBeRemoderatedCampaignIdByBannerId)
                .filterValues(campaignSimple -> campaignSimple.getStatusModerate() == CampaignStatusModerate.NEW)
                .mapValues(CampaignSimple::getId)
                .toMap();

        //Получаем соответствие баннер — все причины отклонения id баннеров
        var modReasonIdsByBannerId = moderationReasonService.getReasonIdsForBannerAndResources(shard, models.keySet());

        return new ListValidationBuilder<>(preValidateResult)
                .checkEach(campaignNotArchived(archivedCampaignIdByBannerId), When.isValid())
                .checkEach(campaignNotDraft(draftCampaignIdByBannerId), When.isValid())
                .checkEach(bannerIsNotDraft(models), When.isValid())
                .checkEach(remoderationIsAllowedForBanner(modReasonIdsByBannerId), When.isValid())
                .getResult();
    }

    private Constraint<ModelChanges<BannerWithSystemFields>, Defect> remoderationIsAllowedForBanner(Map<Long,
            Set<Long>> moderationReasonsById) {
        return mc -> {
            var moderationReasons = moderationReasonsById.get(mc.getId());
            if (moderationReasons == null) {
                return BannerDefects.remoderateBannerWithNoModReasons();
            } else {
                var notAllowedModerationReasons =
                        moderationReasonService.filterReasonsNonAllowedToSelfRemoderate(moderationReasons);
                if (!notAllowedModerationReasons.isEmpty()) {
                    return BannerDefects.remoderateBannerWithNotAllowedModReasons(notAllowedModerationReasons);
                } else {
                    return null;
                }
            }
        };
    }

    /**
     * Проверяет что кампания не архивная.
     */
    private Constraint<ModelChanges<BannerWithSystemFields>, Defect> campaignNotArchived(
            Map<Long, Long> archivedCampaignIdByBannerId) {
        return mc -> Optional.ofNullable(archivedCampaignIdByBannerId.get(mc.getId()))
                .map(BannerDefects::moderateBannerInArchivedCampaign)
                .orElse(null);
    }

    /**
     * Проверяет что кампания не архивная.
     */
    private Constraint<ModelChanges<BannerWithSystemFields>, Defect> campaignNotDraft(
            Map<Long, Long> draftCampaignIdByBannerId) {
        return mc -> Optional.ofNullable(draftCampaignIdByBannerId.get(mc.getId()))
                .map(BannerDefects::remoderateBannerInDraftCampaign)
                .orElse(null);
    }

    /**
     * Проверяет что объявление черновик.
     */
    private Constraint<ModelChanges<BannerWithSystemFields>, Defect> bannerIsDraft(
            Map<Long, BannerWithSystemFields> models) {
        return Constraint.fromPredicate(mc -> models.get(mc.getId()).getStatusModerate() == BannerStatusModerate.NEW,
                moderateNonDraftBanner());
    }

    /**
     * Проверяет что объявление не черновик.
     */
    private Constraint<ModelChanges<BannerWithSystemFields>, Defect> bannerIsNotDraft(
            Map<Long, BannerWithSystemFields> models) {
        return Constraint.fromPredicate(mc -> models.get(mc.getId()).getStatusModerate() != BannerStatusModerate.NEW,
                remoderateDraftBanner());
    }

    /**
     * Проверяет что объявление не архивное.
     */
    private Constraint<ModelChanges<BannerWithSystemFields>, Defect> bannerNotArchived(
            Map<Long, BannerWithSystemFields> models) {
        return Constraint.fromPredicate(mc -> !models.get(mc.getId()).getStatusArchived(),
                moderateArchivedBanner());
    }

    /**
     * Проверяет что у группы объявлений есть условия показа.
     */
    private Constraint<ModelChanges<BannerWithSystemFields>, Defect> adGroupHasShowConditions(
            Set<Long> bannersWithConditions) {
        return Constraint.fromPredicate(mc -> bannersWithConditions.contains(mc.getId()),
                moderateBannerInAdGroupWithoutShowConditions());
    }

    /**
     * Проверяет наличие у клиента/оператора доступа к объявлениям.
     */
    public ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> validateAccess(
            Long operatorUid, ClientId clientId, List<ModelChanges<BannerWithSystemFields>> modelChanges) {
        ListValidationBuilder<ModelChanges<BannerWithSystemFields>, Defect> lvb = ListValidationBuilder.of(
                modelChanges);
        CampaignSubObjectAccessConstraint constraint = campaignSubObjectAccessCheckerFactory
                .newAdsChecker(operatorUid, clientId, mapList(modelChanges, ModelChanges::getId))
                .createAdsValidator(CampaignAccessType.READ_WRITE)
                .getAccessConstraint();
        return lvb
                .checkEach((Constraint<ModelChanges<BannerWithSystemFields>, Defect>) c ->
                        constraint.apply(c.getId()))
                .getResult();
    }

    /**
     * Проверяет наличие у клиента/оператора доступа к перемодерации.
     */
    public ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> validateRemoderationAllowed(
            Long operatorUid, ClientId clientId,
            ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> preValidateResult,
            Set<String> featuresEnabledToClient) {

        var operatorRole = rbacService.getUidRole(operatorUid);
        var clientRole = rbacService.getUidRole(rbacService.getChiefByClientId(clientId));
        boolean remoderationAllowed = featuresEnabledToClient.contains(CLIENT_ALLOWED_TO_REMODERATE.getName());

        return new ListValidationBuilder<>(preValidateResult)
                .check(remoderationAllowedConstraint(operatorRole, clientRole, remoderationAllowed), When.isValid())
                .getResult();
    }

    private Constraint<List<ModelChanges<BannerWithSystemFields>>, Defect> remoderationAllowedConstraint(
            RbacRole operatorRole, RbacRole clientRole, Boolean remoderationAllowed) {
        return mc -> {
            if (!remoderationAllowed(operatorRole, clientRole, remoderationAllowed)) {
                return remoderateBannerIsNotAllowed();
            }
            return null;
        };
    }

    public static Boolean remoderationAllowed(RbacRole operatorRole, RbacRole clientRole, Boolean remoderationAllowed) {
        return Set.of(RbacRole.CLIENT, RbacRole.SUPER).contains(operatorRole)
                && clientRole == RbacRole.CLIENT && remoderationAllowed;
    }
}
