package ru.yandex.direct.core.entity.banner.service.validation;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithInternalInfo;
import ru.yandex.direct.core.entity.banner.model.BannerWithPricePackage;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.service.validation.pricesales.BannerPriceSalesMinusGeoValidator;
import ru.yandex.direct.core.entity.banner.service.validation.pricesales.BannerPriceSalesMinusGeoValidatorFactory;
import ru.yandex.direct.core.entity.banner.type.internal.BannerWithInternalInfoValidatorProvider;
import ru.yandex.direct.core.entity.banner.type.pricepackage.BannerWithPricePackageFullnessModelChangesValidator;
import ru.yandex.direct.core.entity.banner.type.pricepackage.BannerWithPricePackageFullnessValidatorFactory;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.AccessDefectPresets;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignAccessDefects;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessConstraint;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.singleton;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.adNotFound;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.alreadySuspended;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.isNotSuspended;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.resumeArchivedBanner;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.resumeDraftBanner;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.resumeInArchivedCampaign;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.suspendArchivedBanner;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.suspendDraftBanner;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.suspendInArchivedCampaign;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.unsupportedBannerType;
import static ru.yandex.direct.core.validation.defects.RightsDefects.noRights;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.selectList;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

@Service
public class SuspendResumeBannerValidationService {
    private static final CampaignAccessDefects ACCESS_DEFECTS = AccessDefectPresets.AD_ACCESS_DEFECTS.toBuilder()
            .withTypeNotAllowable(adNotFound())
            .withNotVisible(adNotFound())
            .withTypeNotSupported(unsupportedBannerType())
            .withNoRights(objectId -> noRights())
            .build();

    private static final Set<ModelProperty<? extends Model, ?>> ALLOWED_CHANGES =
            singleton(BannerWithSystemFields.STATUS_SHOW);

    private final CampaignRepository campaignRepository;
    private final CampaignSubObjectAccessCheckerFactory accessCheckerFactory;
    private final BannerPriceSalesMinusGeoValidatorFactory priceSalesMinusGeoValidatorFactory;
    private final BannerWithPricePackageFullnessValidatorFactory bannerWithPricePackageFullnessValidatorFactory;
    private final BannerWithInternalInfoValidatorProvider bannerWithInternalInfoValidatorProvider;

    @Autowired
    public SuspendResumeBannerValidationService(
            CampaignRepository campaignRepository,
            CampaignSubObjectAccessCheckerFactory accessCheckerFactory,
            BannerPriceSalesMinusGeoValidatorFactory priceSalesMinusGeoValidatorFactory,
            BannerWithPricePackageFullnessValidatorFactory bannerWithPricePackageFullnessValidatorFactory,
            BannerWithInternalInfoValidatorProvider bannerWithInternalInfoValidatorProvider) {
        this.campaignRepository = campaignRepository;
        this.accessCheckerFactory = accessCheckerFactory;
        this.priceSalesMinusGeoValidatorFactory = priceSalesMinusGeoValidatorFactory;
        this.bannerWithPricePackageFullnessValidatorFactory = bannerWithPricePackageFullnessValidatorFactory;
        this.bannerWithInternalInfoValidatorProvider = bannerWithInternalInfoValidatorProvider;
    }

    /**
     * Предварительная валидация списка баннеров.
     *
     * @param preValidateResult результат предыдущей проверки
     * @param banners           полные модели баннеров по ID
     * @return Результат валидации списка.
     */
    public ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> validateBanners(
            int shard,
            ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> preValidateResult,
            Map<Long, BannerWithSystemFields> banners, boolean resume) {
        Map<Long, Campaign> campaigns = campaignRepository.getCampaignsMap(shard, mapList(banners.values(),
                BannerWithSystemFields::getCampaignId));

        var bannersWithPricePackages = selectList(banners.values(), BannerWithPricePackage.class);

        BannerWithPricePackageFullnessModelChangesValidator priceSalesFullnessValidator =
                bannerWithPricePackageFullnessValidatorFactory.createForSuspendResume(shard, bannersWithPricePackages);

        BannerPriceSalesMinusGeoValidator priceSalesMinusGeoValidator =
                priceSalesMinusGeoValidatorFactory.create(shard, banners, campaigns);

        var bannersWithInternalInfo = EntryStream.of(banners)
                .selectValues(BannerWithInternalInfo.class)
                .toMap();

        var resumeBannerWithInternalExtraInfoValidator =
                bannerWithInternalInfoValidatorProvider.resumeBannerWithInternalExtraInfoListValidator(
                        bannersWithInternalInfo);

        return new ListValidationBuilder<>(preValidateResult)
                .checkEach(campaignIsNotArchived(resume, banners, campaigns), When.isValid())
                .checkEach(adNotArchived(resume, banners), When.isValid())
                .checkEach(adIsNotDraft(resume, banners), When.isValid())
                .checkEachBy(priceSalesFullnessValidator, When.isValidAnd(When.isFalse(resume)))
                .checkEachBy(priceSalesMinusGeoValidator, When.isValidAnd(When.isTrue(resume)))
                .checkBy(resumeBannerWithInternalExtraInfoValidator, When.isValidAnd(When.isTrue(resume)))
                .weakCheckEach(bannerIsNotSuspended(banners), When.isValidAnd(When.isFalse(resume)))
                .weakCheckEach(bannerIsSuspended(banners), When.isValidAnd(When.isTrue(resume)))
                .getResult();
    }

    /**
     * Возвращает условие проверки что баннер не является черновиком
     *
     * @param resume  true если баннер необходимо запустить, false если остановить
     * @param banners полные модели баннеров по ID
     * @return условие для проверки
     */
    private Constraint<ModelChanges<BannerWithSystemFields>, Defect> adIsNotDraft(
            boolean resume,
            Map<Long, BannerWithSystemFields> banners) {

        return Constraint.fromPredicate(
                mc -> banners.containsKey(mc.getId())
                        && banners.get(mc.getId()).getStatusModerate() != BannerStatusModerate.NEW,
                resume ? resumeDraftBanner() : suspendDraftBanner());
    }

    /**
     * Возвращает условие проверки что баннер не заархивирован
     *
     * @param resume  true если баннер необходимо запустить, false если остановить
     * @param banners полные модели баннеров по ID
     * @return условие для проверки
     */
    private Constraint<ModelChanges<BannerWithSystemFields>, Defect> adNotArchived(
            boolean resume,
            Map<Long, BannerWithSystemFields> banners) {

        return Constraint.fromPredicate(
                mc -> banners.containsKey(mc.getId()) && !banners.get(mc.getId()).getStatusArchived(),
                resume ? resumeArchivedBanner() : suspendArchivedBanner());
    }

    /**
     * Возвращает условие проверки что соответствующая баннеру кампания не в архиве.
     *
     * @param resume        true если баннер необходимо запустить, false если остановить
     * @param banners       полные модели баннеров по ID
     * @param campaignsById кампании по ID
     * @return условие для проверки
     */
    private Constraint<ModelChanges<BannerWithSystemFields>, Defect> campaignIsNotArchived(
            boolean resume, Map<Long, BannerWithSystemFields> banners, Map<Long, Campaign> campaignsById) {
        return mc -> {
            if (banners.containsKey(mc.getId())
                    && campaignsById.containsKey(banners.get(mc.getId()).getCampaignId())
                    && !campaignsById.get(banners.get(mc.getId()).getCampaignId()).getStatusArchived()) {
                return null;
            }
            return resume
                    ? resumeInArchivedCampaign(banners.get(mc.getId()).getCampaignId())
                    : suspendInArchivedCampaign(banners.get(mc.getId()).getCampaignId());
        };
    }

    /**
     * Возвращает условие для проверки активности баннера
     *
     * @param banners полные модели баннеров по ID
     * @return условие для проверки
     */
    private Constraint<ModelChanges<BannerWithSystemFields>, Defect> bannerIsNotSuspended(Map<Long,
            BannerWithSystemFields> banners) {
        return Constraint.fromPredicate(
                mc -> banners.containsKey(mc.getId()) && banners.get(mc.getId()).getStatusShow(),
                alreadySuspended());
    }

    /**
     * Возвращает условие для проверки неактивности баннера
     *
     * @param banners полные модели баннеров по ID
     * @return условие для проверки
     */
    private Constraint<ModelChanges<BannerWithSystemFields>, Defect> bannerIsSuspended(Map<Long,
            BannerWithSystemFields> banners) {
        return Constraint.fromPredicate(
                mc -> banners.containsKey(mc.getId()) && !banners.get(mc.getId()).getStatusShow(),
                isNotSuspended());
    }

    public ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> validateChanges(
            ClientId clientId, Long operatorUid, List<ModelChanges<BannerWithSystemFields>> modelChanges) {
        ListValidationBuilder<ModelChanges<BannerWithSystemFields>, Defect> lvb = ListValidationBuilder
                .of(modelChanges, Defect.class);
        lvb.checkEach(onlyStatusShowChanged());
        lvb.weakCheckEach(unique(Comparator.comparing(ModelChanges::getId)), When.isValid());
        validateAccess(clientId, operatorUid, modelChanges, lvb);
        return lvb.getResult();
    }

    private Constraint<ModelChanges<BannerWithSystemFields>, Defect> onlyStatusShowChanged() {
        return Constraint.fromPredicate(mc -> mc.getChangedPropsNames().equals(ALLOWED_CHANGES),
                invalidValue());
    }

    private void validateAccess(ClientId clientId, Long operatorUid,
                                List<ModelChanges<BannerWithSystemFields>> modelChanges,
                                ListValidationBuilder<ModelChanges<BannerWithSystemFields>, Defect> lvb) {
        CampaignSubObjectAccessConstraint constraint = accessCheckerFactory
                .newAdsChecker(operatorUid, clientId, mapList(modelChanges, ModelChanges::getId))
                .createValidator(CampaignAccessType.READ_WRITE, ACCESS_DEFECTS)
                .getAccessConstraint();
        lvb.checkEach((Constraint<ModelChanges<BannerWithSystemFields>, Defect>) c -> constraint.apply(c.getId()),
                When.isValid());
    }
}
