package ru.yandex.direct.core.entity.banner.type.additionalhrefs;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Result;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.BannerAdditionalHref;
import ru.yandex.direct.core.entity.banner.model.BannerAdditionalHrefInternal;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.model.Model;

import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_ADDITIONAL_HREFS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class BannerAdditionalHrefsRepository {

    private final DslContextProvider dslContextProvider;

    public static final JooqMapperWithSupplier<BannerAdditionalHrefInternal> ADDITIONAL_HREFS_INTERNAL_MAPPER =
            JooqMapperWithSupplierBuilder.builder(BannerAdditionalHrefInternal::new)
                    .map(property(BannerAdditionalHrefInternal.ID, BANNER_ADDITIONAL_HREFS.HREF_ID))
                    .map(property(BannerAdditionalHrefInternal.BANNER_ID, BANNER_ADDITIONAL_HREFS.BID))
                    .map(property(BannerAdditionalHrefInternal.INDEX, BANNER_ADDITIONAL_HREFS.INDEX))
                    .map(property(BannerAdditionalHrefInternal.HREF, BANNER_ADDITIONAL_HREFS.HREF))
                    .build();

    private static final JooqMapperWithSupplier<BannerAdditionalHref> ADDITIONAL_HREFS_MAPPER =
            JooqMapperWithSupplierBuilder.builder(BannerAdditionalHref::new)
                    .map(property(BannerAdditionalHref.HREF, BANNER_ADDITIONAL_HREFS.HREF))
                    .build();

    @Autowired
    public BannerAdditionalHrefsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }


    public void deleteAdditionalHrefsForBanners(DSLContext context, List<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return;
        }

        context.delete(BANNER_ADDITIONAL_HREFS)
                .where(BANNER_ADDITIONAL_HREFS.BID.in(bannerIds))
                .execute();
    }

    /**
     * Возвращает список дополнительных ссылок по id баннеров.
     * Если у баннера нет дополнитльных ссылок, то он не попадёт в результат.
     */
    public Map<Long, List<BannerAdditionalHref>> getAdditionalHrefs(int shard, Collection<Long> bannerIds) {
        return getAdditionalHrefs(dslContextProvider.ppc(shard), bannerIds);
    }

    /**
     * Возвращает список дополнительных ссылок по id баннеров.
     * Если у баннера нет дополнитльных ссылок, то он не попадёт в результат.
     */
    public Map<Long, List<BannerAdditionalHref>> getAdditionalHrefs(DSLContext context, Collection<Long> bannerIds) {
        return getAdditionalHrefsWithMapper(context, bannerIds, ADDITIONAL_HREFS_MAPPER);
    }

    public Map<Long, List<BannerAdditionalHrefInternal>> getAdditionalHrefsInternal(DSLContext context,
                                                                                    Collection<Long> bannerIds) {
        return getAdditionalHrefsWithMapper(context, bannerIds, ADDITIONAL_HREFS_INTERNAL_MAPPER);
    }

    private <M extends Model> Map<Long, List<M>> getAdditionalHrefsWithMapper(
            DSLContext context,
            Collection<Long> bannerIds,
            JooqMapperWithSupplier<M> mapper) {
        Map<Long, Result<Record>> bannerIdToAdditionalHrefs = context
                .select(mapper.getFieldsToRead())
                .select(BANNER_ADDITIONAL_HREFS.BID, BANNER_ADDITIONAL_HREFS.INDEX)
                .from(BANNER_ADDITIONAL_HREFS)
                .where(BANNER_ADDITIONAL_HREFS.BID.in(bannerIds))
                .fetchGroups(BANNER_ADDITIONAL_HREFS.BID);

        return EntryStream.of(bannerIdToAdditionalHrefs)
                .mapValues(result -> result
                        .sortAsc(BANNER_ADDITIONAL_HREFS.INDEX)
                        .map(mapper::fromDb))
                .toMap();
    }

}
