package ru.yandex.direct.core.entity.banner.type.additionalhrefs;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.model.BannerAdditionalHref;
import ru.yandex.direct.core.entity.banner.model.BannerAdditionalHrefInternal;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdditionalHrefs;
import ru.yandex.direct.core.entity.banner.repository.AddOrUpdateAndDeleteByKeyContainer;
import ru.yandex.direct.core.entity.banner.repository.AddOrUpdateAndDeleteContainer;
import ru.yandex.direct.core.entity.banner.repository.type.AbstractMultiRowEntityRepositoryTypeSupport;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.banner.model.BannerWithAdditionalHrefs.ADDITIONAL_HREFS;
import static ru.yandex.direct.core.entity.banner.type.additionalhrefs.BannerAdditionalHrefsRepository.ADDITIONAL_HREFS_INTERNAL_MAPPER;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_ADDITIONAL_HREFS;
import static ru.yandex.direct.model.AppliedChanges.isChanged;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@ParametersAreNonnullByDefault
@Component
public class BannerWithAdditionalHrefsRepositoryTypeSupport
        extends AbstractMultiRowEntityRepositoryTypeSupport
        <BannerWithAdditionalHrefs, BannerAdditionalHrefInternal> {

    private final ShardHelper shardHelper;
    private final BannerAdditionalHrefsRepository bannerAdditionalHrefsRepository;

    @Autowired
    public BannerWithAdditionalHrefsRepositoryTypeSupport(DslContextProvider dslContextProvider,
                                                             ShardHelper shardHelper,
                                                             BannerAdditionalHrefsRepository bannerAdditionalHrefsRepository) {
        super(dslContextProvider);
        this.shardHelper = shardHelper;
        this.bannerAdditionalHrefsRepository = bannerAdditionalHrefsRepository;
    }

    @Override
    protected AddOrUpdateAndDeleteContainer<BannerAdditionalHrefInternal> buildAddOrUpdateAndDeleteContainer(
            DSLContext context,
            BannerRepositoryContainer updateParameters,
            Collection<AppliedChanges<BannerWithAdditionalHrefs>> bannersChangesCollection) {

        List<Long> bannerWithChangedAdditionalHrefsIds = filterAndMapList(bannersChangesCollection,
                appliedChanges -> appliedChanges.changed(ADDITIONAL_HREFS),
                appliedChanges -> appliedChanges.getModel().getId());

        Map<Long, List<BannerAdditionalHrefInternal>> currentChangingAdditionalHrefs =
                bannerAdditionalHrefsRepository.getAdditionalHrefsInternal(context,
                        bannerWithChangedAdditionalHrefsIds);

        var builder = AddOrUpdateAndDeleteByKeyContainer.builder(BannerAdditionalHrefInternal::getId)
                .skipNewRowsByEquals();

        StreamEx.of(bannersChangesCollection)
                .filter(isChanged(ADDITIONAL_HREFS))
                .forEach(ac -> {
                    List<BannerAdditionalHrefInternal> oldRows = currentChangingAdditionalHrefs.getOrDefault(
                            ac.getModel().getId(), emptyList());
                    BannerWithAdditionalHrefs banner = ac.getModel();
                    List<BannerAdditionalHrefInternal> newRows = extractRepositoryModels(banner.getId(),
                            banner.getAdditionalHrefs());
                    int minSize = Math.min(oldRows.size(), newRows.size());
                    for (int i = 0; i < minSize; i++) {
                        newRows.get(i).setId(oldRows.get(i).getId());
                    }
                    enrichWithHrefId(newRows.subList(minSize, newRows.size()));
                    builder.addDiff(oldRows, newRows);
                });
        return builder.build();
    }

    @Nonnull
    @Override
    protected List<BannerAdditionalHrefInternal> extractNewRepositoryModels(
            Collection<BannerWithAdditionalHrefs> banners) {
        List<BannerAdditionalHrefInternal> result =
                flatMap(banners, banner -> extractRepositoryModels(banner.getId(), banner.getAdditionalHrefs()));
        enrichWithHrefId(result);
        return result;
    }

    private void enrichWithHrefId(List<BannerAdditionalHrefInternal> additionalHrefs) {
        List<Long> ids = shardHelper.generateBannerAdditionalHrefIds(additionalHrefs.size());
        StreamEx.of(additionalHrefs).zipWith(ids.stream())
                .forKeyValue(BannerAdditionalHrefInternal::setId);
    }

    /**
     * Стоит помнить, что href_id остаётся незаполненным.
     */
    private static List<BannerAdditionalHrefInternal> extractRepositoryModels(
            Long bannerId,
            @Nullable List<BannerAdditionalHref> additionalHrefs) {
        if (additionalHrefs == null) {
            return emptyList();
        }
        return EntryStream.of(additionalHrefs)
                .mapKeyValue((index, additionalHref) -> new BannerAdditionalHrefInternal()
                                .withBannerId(bannerId)
                                .withIndex((long) index)
                                .withHref(additionalHref.getHref())
                        // не можем заполнить id (href_id в таблице)
                )
                .toList();
    }

    @Override
    protected void delete(DSLContext context, Collection<BannerAdditionalHrefInternal> repositoryModelsToDelete) {
        Set<Long> hrefIdToDelete = listToSet(repositoryModelsToDelete, BannerAdditionalHrefInternal::getId);
        context.deleteFrom(BANNER_ADDITIONAL_HREFS)
                .where(BANNER_ADDITIONAL_HREFS.HREF_ID.in(hrefIdToDelete))
                .execute();
    }

    @Override
    protected void addOrUpdate(DSLContext context,
                               Collection<BannerAdditionalHrefInternal> additionalHrefsToAddOrUpdate) {
        new InsertHelper<>(context, BANNER_ADDITIONAL_HREFS)
                .addAll(ADDITIONAL_HREFS_INTERNAL_MAPPER, additionalHrefsToAddOrUpdate)
                .onDuplicateKeyUpdate()
                // не требуется включать сюда BID - он поменяться не может
                .set(BANNER_ADDITIONAL_HREFS.INDEX, MySQLDSL.values(BANNER_ADDITIONAL_HREFS.INDEX))
                .set(BANNER_ADDITIONAL_HREFS.HREF, MySQLDSL.values(BANNER_ADDITIONAL_HREFS.HREF))
                .executeIfRecordsAdded();
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<BannerWithAdditionalHrefs> banners) {
        Set<Long> bannerIds = listToSet(banners, BannerWithAdditionalHrefs::getId);
        Map<Long, List<BannerAdditionalHref>> bannerIdToAdditionalHrefs =
                bannerAdditionalHrefsRepository.getAdditionalHrefs(dslContext, bannerIds);
        banners.forEach(banner ->
                banner.setAdditionalHrefs(nvl(bannerIdToAdditionalHrefs.get(banner.getId()), emptyList())));
    }

    @Override
    public Class<BannerWithAdditionalHrefs> getTypeClass() {
        return BannerWithAdditionalHrefs.class;
    }

}
