package ru.yandex.direct.core.entity.banner.type.aggregatordomain;

import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.BannerWithAggregatorDomain;
import ru.yandex.direct.core.entity.domain.repository.AggregatorDomainsRepository;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.core.entity.banner.model.BannerWithAggregatorDomain.AGGREGATOR_DOMAIN;
import static ru.yandex.direct.core.entity.domain.AggregatorDomainsUtils.extractAggregatorDomainFromHref;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class AggregatorDomainProcessor {

    private final AggregatorDomainsRepository aggregatorDomainsRepository;

    @Autowired
    public AggregatorDomainProcessor(AggregatorDomainsRepository aggregatorDomainsRepository) {
        this.aggregatorDomainsRepository = aggregatorDomainsRepository;
    }

    void addAggregatorDomains(DSLContext dslContext, List<BannerWithAggregatorDomain> banners) {
        aggregatorDomainsRepository.updateAggregatorDomains(dslContext, getAggregatorDomainsByBannerIds(banners));
    }

    void updateAggregatorDomains(DSLContext dslContext,
                                 List<AppliedChanges<BannerWithAggregatorDomain>> appliedChanges) {
        var newAggregatorDomains = getAggregatorDomainsByBannerIds(mapList(appliedChanges, AppliedChanges::getModel));
        appliedChanges.forEach(ac -> ac.modify(AGGREGATOR_DOMAIN,
                newAggregatorDomains.get(ac.getModel().getId())
        ));

        updateChangedAggregatorDomains(dslContext, filterAndMapList(appliedChanges,
                ac -> ac.changedAndNotDeleted(AGGREGATOR_DOMAIN), AppliedChanges::getModel));

        deleteEmptyAggregatorDomains(dslContext, filterAndMapList(appliedChanges,
                ac -> ac.deleted(AGGREGATOR_DOMAIN), AppliedChanges::getModel));
    }

    private void updateChangedAggregatorDomains(DSLContext dslContext,
                                                List<BannerWithAggregatorDomain> bannersWithChangedAggregatorDomain) {
        aggregatorDomainsRepository.updateAggregatorDomains(dslContext,
                listToMap(bannersWithChangedAggregatorDomain,
                        BannerWithAggregatorDomain::getId, BannerWithAggregatorDomain::getAggregatorDomain));
    }

    private void deleteEmptyAggregatorDomains(DSLContext dslContext,
                                              List<BannerWithAggregatorDomain> bannersWithDeletedAggregatorDomain) {
        aggregatorDomainsRepository.deleteAggregatorDomains(dslContext,
                mapList(bannersWithDeletedAggregatorDomain, BannerWithAggregatorDomain::getId));
    }

    private Map<Long, String> getAggregatorDomainsByBannerIds(List<BannerWithAggregatorDomain> banners) {
        return StreamEx.of(banners)
                .mapToEntry(BannerWithAggregatorDomain::getId,
                        banner -> extractAggregatorDomainFromHref(banner.getDomain(), banner.getHref()))
                .filterValues(Objects::nonNull)
                .toMap();
    }
}
