package ru.yandex.direct.core.entity.banner.type.banneradditions;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbschema.ppc.enums.BannersAdditionsAdditionsType;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.lang.Math.min;
import static ru.yandex.direct.dbschema.ppc.Tables.ADDITIONS_ITEM_CALLOUTS;
import static ru.yandex.direct.dbschema.ppc.Tables.ADDITIONS_ITEM_DISCLAIMERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_ADDITIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;

@Repository
public class BannerAdditionsRepository {

    private final DslContextProvider dslContextProvider;

    @Autowired
    public BannerAdditionsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Получить дисклеймеры к объявлениям
     *
     * @param shard     шард для запроса
     * @param bannerIds коллекция id баннеров
     */
    public Map<Long, String> getAdditionDisclaimerByBannerIds(int shard, Collection<Long> bannerIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS.BID, ADDITIONS_ITEM_DISCLAIMERS.DISCLAIMER_TEXT)
                .from(BANNERS_ADDITIONS)
                .join(ADDITIONS_ITEM_DISCLAIMERS).using(ADDITIONS_ITEM_DISCLAIMERS.ADDITIONS_ITEM_ID)
                .join(BANNERS).using(BANNERS.BID)
                .where(BANNERS_ADDITIONS.BID.in(bannerIds))
                .and(BANNERS_ADDITIONS.ADDITIONS_TYPE.eq(BannersAdditionsAdditionsType.disclaimer))
                .fetchMap(BANNERS.BID, ADDITIONS_ITEM_DISCLAIMERS.DISCLAIMER_TEXT);
    }

    /**
     * Получить набор id объявлений в которых используются переданные id дополнений
     *
     * @param shard       шард для запроса
     * @param additionIds коллекция id расширений, которые нужно проверить
     * @return набор id объявлений
     */
    public Set<Long> getBannerIdsByAdditions(int shard, Collection<Long> additionIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS.BID)
                .from(BANNERS_ADDITIONS)
                .join(BANNERS).using(BANNERS.BID)
                .where(BANNERS_ADDITIONS.ADDITIONS_ITEM_ID.in(additionIds))
                .fetchSet(BANNERS.BID);
    }

    /**
     * Получить набор привязанных (используемых) id расширений баннеров
     *
     * @param shard       шард для запроса
     * @param additionIds коллекция id расширений, которые нужно проверить
     * @return набор привязанных уточнений
     */
    public Set<Long> getLinkedBannersAdditions(int shard, Collection<Long> additionIds) {
        return dslContextProvider.ppc(shard)
                .select(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID)
                .from(ADDITIONS_ITEM_CALLOUTS)
                .where(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID.in(additionIds))
                .and(DSL.exists(DSL.selectOne().from(BANNERS_ADDITIONS)
                                .join(BANNERS).using(BANNERS.BID)
                                .join(CAMPAIGNS).using(CAMPAIGNS.CID)
                                .where(BANNERS_ADDITIONS.ADDITIONS_ITEM_ID.eq(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID))
                                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))))
                .fetchSet(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID);
    }

    /**
     * отвязка уточнений клиента от всех баннеров
     *
     * @param context     контекст
     * @param additionIds список id уточнений, которые необходимо отвязать от всех баннеров
     */
    public void deleteBannersAdditionsByAdditionIds(DSLContext context, Collection<Long> additionIds) {
        if (additionIds.isEmpty()) {
            return;
        }

        List<Long> bannerIds = context.selectDistinct(BANNERS_ADDITIONS.BID)
                .from(BANNERS_ADDITIONS)
                .where(BANNERS_ADDITIONS.ADDITIONS_ITEM_ID.in(additionIds))
                .fetch(BANNERS_ADDITIONS.BID);

        if (!bannerIds.isEmpty()) {
            var now = LocalDateTime.now();

            // Баннеров может быть 1,5 миллиона, https://st.yandex-team.ru/DIRECTSUP-34685#5eecb6a6c4de105f643c1e36
            int batchSize = 10_000;
            for (int i = 0; i < bannerIds.size(); i += batchSize) {
                List<Long> batchBannerIds = bannerIds.subList(i, min(bannerIds.size(), i + batchSize));
                context.update(BANNERS)
                        .set(BANNERS.LAST_CHANGE, now)
                        .where(BANNERS.BID.in(batchBannerIds))
                        .execute();
            }
        }

        context.deleteFrom(BANNERS_ADDITIONS)
                .where(BANNERS_ADDITIONS.ADDITIONS_ITEM_ID.in(additionIds))
                .execute();
    }

}
