package ru.yandex.direct.core.entity.banner.type.bannerimage;

import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.banner.model.BannerWithBannerImage;
import ru.yandex.direct.core.entity.banner.model.ImageSize;
import ru.yandex.direct.core.entity.banner.model.ImageType;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.imageNotFound;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.imageSizeInvalid;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.inconsistentStateBannerTypeAndImageType;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

public class BannerWithBannerImageConstraints {
    private BannerWithBannerImageConstraints() {
    }


    /**
     * Проверяет, не было ли смена типа изображения на маленькое, то есть маленькое можно менять на маленькое, все
     * остальные типы на маленькое менять нельзя.
     *
     * @param unmodifiedModels словарь баннеров до применения апдейта (содержат необновленные изображения)
     * @param typedHashes      словарь хэш изображения -> тип, содержащий типы всех изображений (и старых, и новых)
     */
    public static <B extends BannerWithBannerImage>
    Constraint<ModelChanges<B>, Defect> bannerImageTypeIsNotChangedToSmall(
            Map<Long, B> unmodifiedModels,
            Map<String, ImageType> typedHashes) {

        return fromPredicate(after ->
        {
            ImageType typeBefore = getImageType(unmodifiedModels.get(after.getId()), typedHashes);
            ImageType typeAfter = getImageType(after, typedHashes);

            return typeBefore == ImageType.SMALL || typeAfter != ImageType.SMALL;
        }, inconsistentStateBannerTypeAndImageType());
    }

    private static ImageType getImageType(@Nullable BannerWithBannerImage banner,
                                          Map<String, ImageType> typedHashes) {

        return ifNotNull(banner, b -> typedHashes.get(b.getImageHash()));
    }

    private static <B extends BannerWithBannerImage> ImageType getImageType(
            ModelChanges<B> modelChanges,
            Map<String, ImageType> typedHashes) {
        return typedHashes.get(modelChanges.getPropIfChanged(BannerWithBannerImage.IMAGE_HASH));
    }

    /**
     * Проверка вхождения хеша изображения в переданную коллекцию
     */
    public static Constraint<String, Defect> bannerImageExists(Set<String> existHashes) {
        return fromPredicate(existHashes::contains, imageNotFound());
    }

    public static Constraint<String, Defect> isBannerImageTypeValid(ImageType imageType,
                                                                    Set<ImageType> allowedImageTypes) {
        return fromPredicate(imageHash -> allowedImageTypes.contains(imageType),
                inconsistentStateBannerTypeAndImageType());
    }

    public static Constraint<String, Defect> isBannerImageSizeValid(ImageSize imageSize,
                                                                    Set<ImageSize> allowedImageSizes) {
        return fromPredicate(imageHash -> allowedImageSizes.contains(imageSize), imageSizeInvalid());
    }
}
