package ru.yandex.direct.core.entity.banner.type.bannerimage;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jetbrains.annotations.NotNull;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.JoinType;
import org.jooq.Record;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.container.BannersOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithBannerImage;
import ru.yandex.direct.core.entity.banner.model.StatusBannerImageModerate;
import ru.yandex.direct.core.entity.banner.repository.type.AbstractRelatedEntityUpsertRepositoryTypeSupport;
import ru.yandex.direct.core.entity.banner.repository.type.ModifiedPaths;
import ru.yandex.direct.core.entity.image.model.BannerImageFormat;
import ru.yandex.direct.core.grut.api.BannerGrutApi;
import ru.yandex.direct.dbschema.ppc.Tables;
import ru.yandex.direct.dbschema.ppc.enums.BannerImagesStatusshow;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerImagesRecord;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapper.read.JooqReader;
import ru.yandex.direct.jooqmapper.read.JooqReaderBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.multitype.entity.JoinQuery;
import ru.yandex.grut.objects.proto.BannerV2;
import ru.yandex.grut.objects.proto.client.Schema;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.banner.type.bannerimage.BannerWithBannerImageUtils.isBannerImageChanged;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class BannerWithBannerImageRepositoryTypeSupport
        extends AbstractRelatedEntityUpsertRepositoryTypeSupport
        <BannerWithBannerImage, BannerImagesRecord> {

    private final ShardHelper shardHelper;
    private final JooqMapper<BannerWithBannerImage> jooqMapper;
    private final JooqReader<BannerWithBannerImage> idsMapper;

    @Autowired
    public BannerWithBannerImageRepositoryTypeSupport(DslContextProvider dslContextProvider,
                                                      ShardHelper shardHelper) {
        super(dslContextProvider, BANNER_IMAGES.BID);
        this.jooqMapper = createMapper();
        this.idsMapper = createIdsMapper();
        this.shardHelper = shardHelper;
    }

    @Override
    public List<JoinQuery> joinQuery() {
        return List.of(new JoinQuery(
                BANNER_IMAGES,
                JoinType.LEFT_OUTER_JOIN,
                BANNER_IMAGES.BID.eq(BANNERS.BID)
        ));
    }

    /**
     * Джоба которая удаляет старые картинки полностью - RemoveBannerImagesJob
     */
    @Override
    protected void deleteEntities(DSLContext context, BannerRepositoryContainer updateParameters,
                                  Collection<AppliedChanges<BannerWithBannerImage>> appliedChanges) {
        List<Long> bannerIds = mapList(appliedChanges, ac -> ac.getModel().getId());

        //"Удалить" изображения объявлений: проставление им {@code StatusShow = No}.
        context
                .update(Tables.BANNER_IMAGES)
                .set(Tables.BANNER_IMAGES.STATUS_SHOW, BannerImagesStatusshow.No)
                .where(Tables.BANNER_IMAGES.BID.in(bannerIds))
                .execute();
    }


    /**
     * Сохраняет непустые banner.bannerImage в таблицу banners_images
     * (включая генерацию imageId для изображения баннера, если banner.bannerImage непустой).
     * <p>
     * Баннерам уже должны быть присвоены id.
     * <p>
     * Если id уже был задан, то запись обновляется, устанавливается статус показа 'Yes'.
     */
    @Override
    public void upsertEntity(DSLContext context, Collection<BannerWithBannerImage> banners) {
        generateAndSetNewIds(banners);

        InsertHelper<BannerImagesRecord> insertHelper = new InsertHelper<>(context, BANNER_IMAGES)
                .addAll(jooqMapper, banners);

        insertHelper.onDuplicateKeyUpdate()
                .set(BANNER_IMAGES.STATUS_SHOW, BannerImagesStatusshow.Yes)
                .set(BANNER_IMAGES.STATUS_MODERATE, MySQLDSL.values(BANNER_IMAGES.STATUS_MODERATE))
                .set(BANNER_IMAGES.IMAGE_HASH, MySQLDSL.values(BANNER_IMAGES.IMAGE_HASH));

        insertHelper.executeIfRecordsAdded();
    }

    private void generateAndSetNewIds(Collection<BannerWithBannerImage> bannersWithImage) {
        List<BannerWithBannerImage> bannersWithImageWithNoId =
                filterList(bannersWithImage, b -> b.getImageId() == null);
        Iterator<Long> bannerImageIds = generateBannerImageIds(bannersWithImageWithNoId).iterator();

        bannersWithImageWithNoId.forEach(banner -> banner.setImageId(bannerImageIds.next()));
    }

    /**
     * Сгенерировать новые id для bannerImage.
     * Используется общий генератор с баннером по аналогии с perl - реализацией.
     *
     * @param bannersWithImage баннеры с картинкой
     */
    private <B extends BannerWithBannerImage> List<Long> generateBannerImageIds(List<B> bannersWithImage) {
        return shardHelper.generateBannerIdsByBids(mapList(bannersWithImage, B::getId));
    }

    @Override
    protected boolean isAddEntity(BannerWithBannerImage model) {
        return model.getImageHash() != null;
    }

    @Override
    protected boolean isDeleteEntity(AppliedChanges<BannerWithBannerImage> appliedChange) {
        return appliedChange.deleted(BannerWithBannerImage.IMAGE_HASH);
    }

    @Override
    protected boolean isUpsertEntity(AppliedChanges<BannerWithBannerImage> ac) {
        return !isDeleteEntity(ac) && isBannerImageChanged(ac);
    }

    @Override
    public Class<BannerWithBannerImage> getTypeClass() {
        return BannerWithBannerImage.class;
    }

    private static JooqMapper<BannerWithBannerImage> createMapper() {
        return JooqMapperBuilder.<BannerWithBannerImage>builder()
                .writeField(BANNER_IMAGES.BID, fromProperty(BannerWithBannerImage.ID))
                .map(property(BannerWithBannerImage.IMAGE_ID, BANNER_IMAGES.IMAGE_ID))
                .map(property(BannerWithBannerImage.IMAGE_HASH, BANNER_IMAGES.IMAGE_HASH))
                .map(property(BannerWithBannerImage.IMAGE_NAME, BANNER_IMAGES.NAME))
                .map(property(BannerWithBannerImage.IMAGE_BS_BANNER_ID, BANNER_IMAGES.BANNER_ID))
                .map(convertibleProperty(BannerWithBannerImage.IMAGE_STATUS_MODERATE, BANNER_IMAGES.STATUS_MODERATE,
                        StatusBannerImageModerate::fromSource, StatusBannerImageModerate::toSource))
                .map(convertibleProperty(BannerWithBannerImage.IMAGE_STATUS_SHOW, BANNER_IMAGES.STATUS_SHOW,
                        BannerWithBannerImageRepositoryTypeSupport::imageStatusShowFromDb,
                        BannerWithBannerImageRepositoryTypeSupport::imageStatusShowToDb))
                .map(property(BannerWithBannerImage.IMAGE_DATE_ADDED, BANNER_IMAGES.DATE_ADDED))
                .map(convertibleProperty(BannerWithBannerImage.OPTS, BANNER_IMAGES.OPTS,
                        BannerImageConverterKt::optsFromDb, BannerImageConverterKt::optsToDb))
                .build();
    }

    private static JooqReader<BannerWithBannerImage> createIdsMapper() {
        return JooqReaderBuilder.<BannerWithBannerImage>builder()
                .readProperty(BannerWithBannerImage.DELETED_IMAGE_ID, fromField(BANNER_IMAGES.IMAGE_ID))
                .readProperty(BannerWithBannerImage.DELETED_IMAGE_BS_BANNER_ID, fromField(BANNER_IMAGES.BANNER_ID))
                .build();
    }

    private static Boolean imageStatusShowFromDb(@Nullable BannerImagesStatusshow statusShow) {
        return statusShow != null ? statusShow == BannerImagesStatusshow.Yes : null;
    }

    private static BannerImagesStatusshow imageStatusShowToDb(@Nullable Boolean statusShow) {
        if (statusShow == null) {
            return null;
        }
        return statusShow ? BannerImagesStatusshow.Yes : BannerImagesStatusshow.No;
    }

    /**
     * Для обратной совместимости с текущим кодом сохраняем поведение,
     * при котором все (старые) поля BannerWithBannerImage остаются незаполненными для картинок со statusShow=No.
     * Заполняем для таких картинок два новых поля: deletedImageId, deletedImageBsBannerId
     */
    @Override
    public final void fillFromRecord(BannerWithBannerImage model, Record record) {
        if (BannerImagesStatusshow.Yes.equals(record.get(BANNER_IMAGES.STATUS_SHOW))) {
            jooqMapper.fromDb(record, model);
        } else {
            idsMapper.fromDb(record, model);
        }
    }

    @Override
    public Collection<Field<?>> getFields() {
        return jooqMapper.getFieldsToRead();
    }


    @Override
    public Set<ModelProperty<? super BannerWithBannerImage, ?>> getGrutSupportedProperties() {
        // На самом деле реально в грут едут только изменения image_hash, остальное (пока?) не интересно
        return Set.of(
                BannerWithBannerImage.IMAGE_ID,
                BannerWithBannerImage.IMAGE_HASH,
                BannerWithBannerImage.IMAGE_NAME,
                BannerWithBannerImage.IMAGE_BS_BANNER_ID,
                BannerWithBannerImage.IMAGE_STATUS_MODERATE,
                BannerWithBannerImage.IMAGE_STATUS_SHOW,
                BannerWithBannerImage.IMAGE_DATE_ADDED
        );
    }

    @Override
    public Map<Long, ModifiedPaths> applyToGrutObjects(@NotNull Map<Long, Schema.TBannerV2.Builder> bannerBuilders,
                                                       @NotNull Collection<AppliedChanges<BannerWithBannerImage>> appliedChangesList,
                                                       @NotNull BannersOperationContainer operationContainer) {
        Map<Long, ModifiedPaths> modifiedPathsMap = new HashMap<>();
        for (AppliedChanges<BannerWithBannerImage> appliedChanges : appliedChangesList) {
            if (appliedChanges.getPropertiesForUpdate().contains(BannerWithBannerImage.IMAGE_HASH)) {
                Long id = appliedChanges.getModel().getId();
                Schema.TBannerV2.Builder bannerBuilder = bannerBuilders.get(id);
                ModifiedPaths modifiedPaths;
                String newValue = appliedChanges.getNewValue(BannerWithBannerImage.IMAGE_HASH);
                if (newValue != null) {
                    BannerImageFormat bannerImageFormat = operationContainer.getBannerImageFormats().get(newValue);
                    BannerV2.TBannerV2Spec.TImage tImage = BannerGrutApi.Companion.toTImage(
                            appliedChanges.getModel(), bannerImageFormat);
                    bannerBuilder.getSpecBuilder().clearImages();
                    bannerBuilder.getSpecBuilder().addImages(tImage);

                    modifiedPaths = new ModifiedPaths(Set.of("/spec/images"), emptySet());
                } else {
                    modifiedPaths = new ModifiedPaths(emptySet(), Set.of("/spec/images"));
                }
                modifiedPathsMap.put(id, modifiedPaths);
            }
        }
        return modifiedPathsMap;
    }
}
