package ru.yandex.direct.core.entity.banner.type.bannerimage;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannerAdditionalActionsContainer;
import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainer;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerImageOpts;
import ru.yandex.direct.core.entity.banner.model.BannerWithBannerImage;
import ru.yandex.direct.core.entity.banner.model.BannerWithCampaignId;
import ru.yandex.direct.core.entity.banner.service.type.update.AbstractBannerUpdateOperationTypeSupport;
import ru.yandex.direct.core.entity.banner.type.image.BannerImageRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.banner.type.bannerimage.BannerWithBannerImageUtils.isBannerImageChanged;
import static ru.yandex.direct.feature.FeatureName.SINGLE_IMAGE_AD_TO_BS;
import static ru.yandex.direct.feature.FeatureName.SINGLE_IMAGE_AD_TO_BS_AFTER_UPDATE_IMAGE;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class BannerWithBannerImageUpdateOperationTypeSupport
        extends AbstractBannerUpdateOperationTypeSupport<BannerWithBannerImage> {

    private final BannerImageRepository bannerImageRepository;
    private final FeatureService featureService;

    @Autowired
    public BannerWithBannerImageUpdateOperationTypeSupport(BannerImageRepository bannerImageRepository,
                                                           FeatureService featureService) {
        this.bannerImageRepository = bannerImageRepository;
        this.featureService = featureService;
    }


    @Override
    public Class<BannerWithBannerImage> getTypeClass() {
        return BannerWithBannerImage.class;
    }

    @Override
    public void beforeExecution(BannersUpdateOperationContainer updateContainer,
                                List<AppliedChanges<BannerWithBannerImage>> changesList) {
        boolean singleAdToBSEnabled = featureService
                .isEnabledForClientId(updateContainer.getClientId(), SINGLE_IMAGE_AD_TO_BS);
        // Т.к. выставление флага при добавлении картинки уже было поддержано ранее, то обновление прячем за фичей
        // для предварительной проверки в проде
        boolean singleAdToBSAfterUpdateImage = featureService
                .isEnabledForClientId(updateContainer.getClientId(), SINGLE_IMAGE_AD_TO_BS_AFTER_UPDATE_IMAGE);
        if (!singleAdToBSEnabled || !singleAdToBSAfterUpdateImage) {
            return;
        }

        List<BannerWithBannerImage> banners = mapList(changesList, AppliedChanges::getModel);
        Map<Long, BannerWithBannerImage> bidToBanner = StreamEx.of(banners)
                .mapToEntry(Banner::getId, Function.identity())
                .toMap();
        Set<Long> bidsWithVeryFirstImage = StreamEx.of(banners)
                .filter(banner -> banner.getImageHash() != null)
                .filter(banner -> banner.getImageId() == null)
                .map(BannerWithCampaignId::getId)
                .toSet();
        if (bidsWithVeryFirstImage.isEmpty()) {
            return;
        }

        // Проверяем что в базе действительно еще не было картинок и баннеров
        var bidsThatHadImagesBefore = bannerImageRepository
                .getBannerIdsWithImages(updateContainer.getShard(), bidsWithVeryFirstImage);
        var bannersWithVeryFirstImage = StreamEx.of(bidsWithVeryFirstImage)
                .filter(bid -> !bidsThatHadImagesBefore.contains(bid))
                .map(bidToBanner::get)
                .toList();

        for (BannerWithBannerImage banner : bannersWithVeryFirstImage) {
            Set<BannerImageOpts> opts = banner.getOpts() != null
                    ? new HashSet<>(banner.getOpts())
                    : new HashSet<>();
            opts.add(BannerImageOpts.SINGLE_AD_TO_BS);
            banner.withOpts(opts);
        }
    }

    @Override
    public boolean needBsResync(AppliedChanges<BannerWithBannerImage> appliedChanges) {
        return isBannerImageChanged(appliedChanges);
    }

    @Override
    public boolean needLastChangeReset(AppliedChanges<BannerWithBannerImage> appliedChanges) {
        return isBannerImageChanged(appliedChanges);
    }

    @Override
    public void addToAdditionalActionsContainer(
            BannerAdditionalActionsContainer additionalActionsContainer,
            BannersUpdateOperationContainer updateContainer,
            List<AppliedChanges<BannerWithBannerImage>> appliedChanges) {
        sendAdGroupsToSyncWithBannerImages(additionalActionsContainer, appliedChanges);
    }

    /**
     * Для всех баннеров, у которых добавлено изображение.
     * Все группы с такими баннерами отправляются на пересинхронизацию в БК.
     *
     * @param appliedChanges изменения баннеров
     */
    private void sendAdGroupsToSyncWithBannerImages(
            BannerAdditionalActionsContainer additionalActionsContainer,
            Collection<AppliedChanges<BannerWithBannerImage>> appliedChanges) {
        Set<Long> adGroupsToSync = appliedChanges.stream()
                .filter(changes -> changes.assigned(BannerWithBannerImage.IMAGE_HASH))
                .map(changes -> changes.getModel().getAdGroupId())
                .collect(toSet());
        additionalActionsContainer.addAdGroupsIdsForBSResync(adGroupsToSync);
    }
}
