package ru.yandex.direct.core.entity.banner.type.bannerimage.moderation;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannerAdditionalActionsContainer;
import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainer;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithBannerImageModeration;
import ru.yandex.direct.core.entity.banner.model.StatusBannerImageModerate;
import ru.yandex.direct.core.entity.moderation.service.ModerationService;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.core.entity.banner.model.BannerWithBannerImage.IMAGE_BS_BANNER_ID;
import static ru.yandex.direct.core.entity.banner.model.BannerWithBannerImage.IMAGE_DATE_ADDED;
import static ru.yandex.direct.core.entity.banner.model.BannerWithBannerImage.IMAGE_HASH;
import static ru.yandex.direct.core.entity.banner.model.BannerWithBannerImage.IMAGE_STATUS_MODERATE;
import static ru.yandex.direct.core.entity.banner.model.BannerWithBannerImage.IMAGE_STATUS_SHOW;
import static ru.yandex.direct.core.entity.banner.model.BannerWithModerationStatuses.STATUS_MODERATE;
import static ru.yandex.direct.core.entity.banner.service.BannerModerationUtils.bannerBecameDraft;
import static ru.yandex.direct.core.entity.banner.service.BannerUtils.getCampaignIdToBannerIds;
import static ru.yandex.direct.core.entity.banner.type.bannerimage.BannerWithBannerImageUtils.isBannerImageChanged;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class DefaultBannerWithBannerImageModerationProcessor implements BannerWithBannerImageModerationProcessor {

    private final ModerationService moderationService;

    @Autowired
    public DefaultBannerWithBannerImageModerationProcessor(ModerationService moderationService) {
        this.moderationService = moderationService;
    }

    @Override
    public Class<BannerWithBannerImageModeration> getProcessedClass() {
        return BannerWithBannerImageModeration.class;
    }

    @Override
    public void process(
            DSLContext dsl,
            BannerAdditionalActionsContainer additionalActionsContainer, BannersUpdateOperationContainer container,
            List<AppliedChanges<BannerWithBannerImageModeration>> appliedChanges) {

        List<BannerWithBannerImageModeration> toDeleteFromModeration = new ArrayList<>();

        for (AppliedChanges<BannerWithBannerImageModeration> bannerChanges : appliedChanges) {
            if (!bannerChanges.hasActuallyChangedProps() && container.getModerationMode().isDefault()) {
                continue;
            }

            String oldImageHash = bannerChanges.getOldValue(IMAGE_HASH);
            String newImageHash = bannerChanges.getNewValue(IMAGE_HASH);
            boolean imageChanged = isBannerImageChanged(bannerChanges);
            BannerStatusModerate newBannerStatusModerate = bannerChanges.getNewValue(STATUS_MODERATE);

            // newImageHash != null может означать, что
            // либо у баннера будет новая картинка (старая картинка может как быть так и не быть, не важно),
            // либо изменений в картинке нет, но в бд точно есть старая картинка

            if (newImageHash != null) {
                if (newBannerStatusModerate == BannerStatusModerate.NEW) {
                    fillDefaultFieldsAndSetImageStatusModerate(bannerChanges, StatusBannerImageModerate.NEW);
                } else if (newBannerStatusModerate == BannerStatusModerate.READY || imageChanged) {
                    fillDefaultFieldsAndSetImageStatusModerate(bannerChanges, StatusBannerImageModerate.READY);
                }
            }

            if (needToClearModerationData(bannerChanges, oldImageHash, newImageHash)) {
                toDeleteFromModeration.add(bannerChanges.getModel());
            }

        }

        deleteFromModeration(dsl, container, toDeleteFromModeration);
    }


    /**
     * Нужна ли очистка данных модерации изображений
     */
    private boolean needToClearModerationData(AppliedChanges<BannerWithBannerImageModeration> bannerChanges,
                                              String oldImageHash,
                                              String newImageHash) {

        if (bannerChanges.deleted(IMAGE_HASH)) {
            return true;
        }

        boolean imageWasNotDraft = bannerChanges.getOldValue(IMAGE_STATUS_MODERATE) != StatusBannerImageModerate.NEW;
        // Если баннер является черновиком, а картинка не была черновиком, то нужно удалить её из модерации
        return oldImageHash != null
                && newImageHash != null
                && bannerBecameDraft(bannerChanges)
                && imageWasNotDraft;
    }

    private void deleteFromModeration(DSLContext dsl,
                                      BannersUpdateOperationContainer container,
                                      List<BannerWithBannerImageModeration> toDeleteFromModeration) {

        List<Long> bannerIdsToModeration = mapList(toDeleteFromModeration, Banner::getId);
        Map<Long, List<Long>> campaignIdToBannerIdsWithDeletedImages =
                getCampaignIdToBannerIds(toDeleteFromModeration, banner -> banner.getImageHash() == null);
        moderationService.clearImagesModeration(dsl, container.getClientId(),
                bannerIdsToModeration, campaignIdToBannerIdsWithDeletedImages);
    }

    private void fillDefaultFieldsAndSetImageStatusModerate(AppliedChanges<BannerWithBannerImageModeration> bannerChanges,
                                                            StatusBannerImageModerate status) {

        bannerChanges.modify(IMAGE_STATUS_MODERATE, status);

        // если картинки у баннера не было
        if (bannerChanges.getNewValue(IMAGE_STATUS_SHOW) == null) {
            bannerChanges.modify(IMAGE_STATUS_SHOW, true);
        }

        if (bannerChanges.getNewValue(IMAGE_DATE_ADDED) == null) {
            bannerChanges.modify(IMAGE_DATE_ADDED, LocalDateTime.now());
        }

        if (bannerChanges.getNewValue(IMAGE_BS_BANNER_ID) == null) {
            bannerChanges.modify(IMAGE_BS_BANNER_ID, 0L);
        }
    }
}
