package ru.yandex.direct.core.entity.banner.type.body;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;
import ru.yandex.direct.core.entity.banner.container.BannersOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithBody;
import ru.yandex.direct.core.entity.banner.model.BannerWithFixedBody;
import ru.yandex.direct.core.entity.banner.model.ContentPromotionBanner;
import ru.yandex.direct.core.entity.banner.model.CpcVideoBanner;
import ru.yandex.direct.core.entity.banner.model.CpmBanner;
import ru.yandex.direct.core.entity.banner.model.DynamicBanner;
import ru.yandex.direct.core.entity.banner.model.ImageBanner;
import ru.yandex.direct.core.entity.banner.model.MobileAppBanner;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.service.validation.BannerTextValidator;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstants.MAX_NUMBER_OF_NARROW_CHARACTERS;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstraints.charsAreAllowed;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstraints.stringIsNotBlank;
import static ru.yandex.direct.core.entity.banner.type.body.BannerWithBodyConstants.MAX_LENGTH_BODY;
import static ru.yandex.direct.core.entity.banner.type.body.BannerWithBodyConstants.MAX_LENGTH_BODY_WORD;
import static ru.yandex.direct.core.entity.banner.type.body.BannerWithBodyConstants.MAX_LENGTH_CONTENT_PROMOTION_BODY;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.util.ConstraintsValidator.emptyValidator;
import static ru.yandex.direct.validation.util.ConstraintsValidator.oneConstraintValidator;

@Component
public class BannerWithBodyValidatorProvider {

    public Validator<BannerWithBody, Defect> bannerWithBodyValidator(BannersOperationContainer container) {
        return bannerWithBody -> {
            ModelItemValidationBuilder<BannerWithBody> ivb = ModelItemValidationBuilder.of(bannerWithBody);
            ivb.item(BannerWithBody.BODY)
                    .checkBy(bodyValidator(bannerWithBody, container));
            return ivb.getResult();
        };
    }

    private Validator<String, Defect> bodyValidator(BannerWithBody banner, BannersOperationContainer container) {

        if (banner instanceof TextBanner) {
            return BannerTextValidator.builder(MAX_LENGTH_BODY, MAX_LENGTH_BODY_WORD)
                    .withMaxNumberOfNarrowCharacters(MAX_NUMBER_OF_NARROW_CHARACTERS)
                    .build();
        }

        if (banner instanceof CpmBanner || banner instanceof ImageBanner) {
            // если это видео на главной, то для неё специфичный лимит на заголовок и заголовок обязательный
            PricePackage pricePackage = container.getPricePackage(banner);
            boolean isControlByPackage = pricePackage != null && pricePackage.isFrontpageVideoPackage();
            return banner.getBody() == null && !isControlByPackage ? emptyValidator() : BannerTextValidator
                    .builder(MAX_LENGTH_BODY, MAX_LENGTH_BODY_WORD)
                    .withMaxNumberOfNarrowCharacters(MAX_NUMBER_OF_NARROW_CHARACTERS)
                    .build();
        }

        if (banner instanceof DynamicBanner) {
            return BannerTextValidator.builder(MAX_LENGTH_BODY, MAX_LENGTH_BODY_WORD)
                    .withMaxNumberOfNarrowCharacters(MAX_NUMBER_OF_NARROW_CHARACTERS)
                    .withTemplateDisallowed()
                    .build();
        }

        if (banner instanceof MobileAppBanner) {
            return BannerTextValidator.builder(MAX_LENGTH_BODY, MAX_LENGTH_BODY_WORD).build();
        }

        if (banner instanceof ContentPromotionBanner) {
            ContentPromotionAdgroupType contentPromotionAdgroupType = container.getContentPromotionAdGroupType(banner);

            // невозможно провалидировать тело, когда тип контента группы неизвестен.
            // эта проблема должна быть поймана в другом месте.
            if (contentPromotionAdgroupType == null) {
                return emptyValidator();
            }

            switch (contentPromotionAdgroupType) {
                case VIDEO:
                    return BannerTextValidator.builder(MAX_LENGTH_CONTENT_PROMOTION_BODY, MAX_LENGTH_BODY_WORD)
                            .withMaxNumberOfNarrowCharacters(MAX_NUMBER_OF_NARROW_CHARACTERS)
                            .build();
                case EDA:
                    // В баннере продвижения Я.еды тело нигде не показываем, поэтому валидируем по-минимуму:
                    // чтоб был, чтоб был не очень длинным и не содержал запрещенных символов
                    return getWeakenedBodyValidatorContentPromotion();
            }
        }

        // в этом типе баннера тело является системным полем и поэтому не валидируется
        if (banner instanceof CpcVideoBanner || banner instanceof BannerWithFixedBody) {
            return emptyValidator();
        }

        // По умолчанию запрещаем устанавливать тело баннера.
        // Таким образом, если для нового типа баннера забыли включить валидацию тела,
        // это обнаружится самым простым тестом.
        return oneConstraintValidator(isNull());
    }

    //на непустоту не проверяем, так как body у Еды опционален
    private Validator<String, Defect> getWeakenedBodyValidatorContentPromotion() {
        return title -> {
            ItemValidationBuilder<String, Defect> vb = ItemValidationBuilder.of(title);
            vb
                    .check(stringIsNotBlank())
                    .check(charsAreAllowed())
                    .check(maxStringLength(MAX_LENGTH_CONTENT_PROMOTION_BODY));
            return vb.getResult();
        };
    }
}
