package ru.yandex.direct.core.entity.banner.type.callouts;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.model.AdditionType;
import ru.yandex.direct.core.entity.banner.model.BannerAddition;
import ru.yandex.direct.core.entity.banner.model.BannerWithCallouts;
import ru.yandex.direct.core.entity.banner.repository.AddOrUpdateAndDeleteByKeyContainer;
import ru.yandex.direct.core.entity.banner.repository.AddOrUpdateAndDeleteContainer;
import ru.yandex.direct.core.entity.banner.repository.type.AbstractMultiRowEntityRepositoryTypeSupport;
import ru.yandex.direct.dbschema.ppc.enums.BannersAdditionsAdditionsType;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static ru.yandex.direct.core.entity.banner.model.BannerWithCallouts.CALLOUT_IDS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_ADDITIONS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.model.AppliedChanges.isChanged;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Component
public class BannerWithCalloutsRepositoryTypeSupport
        extends AbstractMultiRowEntityRepositoryTypeSupport
        <BannerWithCallouts, BannerAddition> {

    private final JooqMapperWithSupplier<BannerAddition> bannerAdditionsMapper;

    @Autowired
    public BannerWithCalloutsRepositoryTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider);
        this.bannerAdditionsMapper = createBannerAdditionMapper();
    }

    @Override
    protected AddOrUpdateAndDeleteContainer<BannerAddition> buildAddOrUpdateAndDeleteContainer(
            DSLContext context,
            BannerRepositoryContainer updateParameters,
            Collection<AppliedChanges<BannerWithCallouts>> bannersChangesCollection) {
        var builder = AddOrUpdateAndDeleteByKeyContainer.builder(BannerAddition::getId)
                .skipNewRowsByEquals();

        StreamEx.of(bannersChangesCollection)
                .filter(isChanged(CALLOUT_IDS))
                .forEach(ac -> builder.addDiff(extractOldRepositoryModels(ac),
                        extractNewRepositoryModels(singletonList(ac.getModel()))));

        return builder.build();
    }

    @Nonnull
    @Override
    protected List<BannerAddition> extractNewRepositoryModels(Collection<BannerWithCallouts> banners) {
        return flatMap(banners, banner -> extractRepositoryModels(banner.getId(), banner.getCalloutIds()));
    }

    @Override
    protected void delete(DSLContext context, Collection<BannerAddition> calloutsToDelete) {
        Condition condition = StreamEx.of(calloutsToDelete)
                .map(callout -> BANNERS_ADDITIONS.BID.eq(callout.getBannerId())
                        .and(BANNERS_ADDITIONS.ADDITIONS_ITEM_ID.in(callout.getId())))
                .reduce(Condition::or)
                .orElse(DSL.falseCondition());
        context.deleteFrom(BANNERS_ADDITIONS)
                .where(condition)
                .execute();
    }

    @Override
    protected void addOrUpdate(DSLContext context, Collection<BannerAddition> calloutsToAddOrUpdate) {
        new InsertHelper<>(context, BANNERS_ADDITIONS)
                .addAll(bannerAdditionsMapper, calloutsToAddOrUpdate)
                .onDuplicateKeyUpdate()
                .set(BANNERS_ADDITIONS.SEQUENCE_NUM, MySQLDSL.values(BANNERS_ADDITIONS.SEQUENCE_NUM))
                .executeIfRecordsAdded();
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<BannerWithCallouts> banners) {
        Set<Long> bannerIds = listToSet(banners, BannerWithCallouts::getId);
        Map<Long, List<Long>> bannerIdToCalloutIds = getCalloutIdsByBannerIds(dslContext, bannerIds);
        banners.forEach(banner ->
                banner.setCalloutIds(defaultIfNull(bannerIdToCalloutIds.get(banner.getId()), emptyList())));
    }

    @Override
    public Class<BannerWithCallouts> getTypeClass() {
        return BannerWithCallouts.class;
    }

    private Collection<BannerAddition> extractOldRepositoryModels(
            AppliedChanges<BannerWithCallouts> bannerChanges) {
        return extractRepositoryModels(bannerChanges.getModel().getId(), bannerChanges.getOldValue(CALLOUT_IDS));
    }

    private Collection<BannerAddition> extractRepositoryModels(Long bannerId, @Nullable List<Long> calloutIds) {
        if (isEmpty(calloutIds)) {
            return emptyList();
        }
        //в уточнениях счет начинается с единицы
        long orderNum = 1;
        List<BannerAddition> bannerAdditions = new ArrayList<>();
        for (Long calloutId : calloutIds) {
            bannerAdditions.add(new BannerAddition()
                    .withId(calloutId)
                    .withBannerId(bannerId)
                    .withAdditionType(AdditionType.CALLOUT)
                    .withSequenceNum(orderNum++));
        }
        return bannerAdditions;
    }

    private Map<Long, List<Long>> getCalloutIdsByBannerIds(DSLContext dslContext, Collection<Long> bannerIds) {
        Function<List<BannerAddition>, List<BannerAddition>> sortBySequenceNum =
                bannerAdditions -> bannerAdditions.stream()
                        .sorted(Comparator.comparing(BannerAddition::getSequenceNum))
                        .collect(toList());
        List<BannerAddition> calloutsResult = dslContext
                .select(BANNERS_ADDITIONS.BID, BANNERS_ADDITIONS.ADDITIONS_ITEM_ID, BANNERS_ADDITIONS.SEQUENCE_NUM)
                .from(BANNERS_ADDITIONS)
                .where(BANNERS_ADDITIONS.BID.in(bannerIds))
                .and(BANNERS_ADDITIONS.ADDITIONS_TYPE.eq(BannersAdditionsAdditionsType.callout))
                .fetch(bannerAdditionsMapper::fromDb);
        return StreamEx.of(calloutsResult)
                .mapToEntry(BannerAddition::getBannerId, identity())
                .collapseKeys(collectingAndThen(toList(), sortBySequenceNum))
                .mapValues(bannersAdditions -> mapList(bannersAdditions, BannerAddition::getId))
                .toMap();
    }

    private static JooqMapperWithSupplier<BannerAddition> createBannerAdditionMapper() {
        return JooqMapperWithSupplierBuilder.builder(BannerAddition::new)
                .map(property(BannerAddition.ID, BANNERS_ADDITIONS.ADDITIONS_ITEM_ID))
                .map(property(BannerAddition.BANNER_ID, BANNERS_ADDITIONS.BID))
                .map(property(BannerAddition.SEQUENCE_NUM, BANNERS_ADDITIONS.SEQUENCE_NUM))
                .map(convertibleProperty(BannerAddition.ADDITION_TYPE, BANNERS_ADDITIONS.ADDITIONS_TYPE,
                        AdditionType::fromSource, AdditionType::toSource))
                .build();
    }

}
