package ru.yandex.direct.core.entity.banner.type.callouts;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.addition.callout.repository.CalloutRepository;
import ru.yandex.direct.core.entity.banner.container.BannersOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithCallouts;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.adExtensionNotFound;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.duplicatedAdExtensionId;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.invalidIdInArrayElement;
import static ru.yandex.direct.core.entity.banner.type.callouts.BannerWithCalloutsConstants.MAX_CALLOUTS_COUNT_ON_BANNER;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@Component
public class BannerWithCalloutsValidatorProvider {

    CalloutRepository calloutRepository;

    @Autowired
    public BannerWithCalloutsValidatorProvider(CalloutRepository calloutRepository) {
        this.calloutRepository = calloutRepository;
    }

    public Validator<BannerWithCallouts, Defect> validator(
            BannersOperationContainer container, Function<Integer, Defect> maxListSizeDefectProvider,
            List<BannerWithCallouts> banners) {
        Set<Long> accessibleCallouts = getAccessibleCallouts(container.getShard(), container.getClientId(), banners);
        return bannerWithCallouts -> {
            ModelItemValidationBuilder<BannerWithCallouts> ivb =
                    ModelItemValidationBuilder.of(bannerWithCallouts);
            ivb.item(BannerWithCallouts.CALLOUT_IDS)
                    .checkBy(calloutIdsValidator(maxListSizeDefectProvider, accessibleCallouts), When.notNull());
            return ivb.getResult();
        };
    }

    public static Validator<List<Long>, Defect> calloutIdsValidator(
            Function<Integer, Defect> maxListSizeDefectProvider, Set<Long> accessibleCallouts) {
        return calloutIds ->
        {
            ListValidationBuilder<Long, Defect> lvb = ListValidationBuilder.of(calloutIds);
            return lvb
                    .check(maxListSize(MAX_CALLOUTS_COUNT_ON_BANNER),
                            maxListSizeDefectProvider.apply(MAX_CALLOUTS_COUNT_ON_BANNER))
                    .checkEach(notNull())
                    .checkEach(validId(), invalidIdInArrayElement(), When.isValid())
                    // проверяем hasErrors(), а не When.isValid(), чтобы запустить проверку accessibleCallouts, даже
                    // если какие-то из них null или невалидны
                    .checkBy(accessibleCalloutsValidator(accessibleCallouts), When.isTrue(!lvb.getResult().hasErrors()))
                    .getResult();
        };
    }

    private static Validator<List<Long>, Defect> accessibleCalloutsValidator(Set<Long> accessibleCallouts) {
        // немного ручного управления validation result-ом.
        // здесь нужно выдавать уникальные (по id) ошибки, см. юнит-тест
        return calloutIds -> {
            var lvb = ListValidationBuilder.<Long, Defect>of(calloutIds);
            Map<Long, Defect> defects = new HashMap<>();
            Set<Long> distinctIds = new HashSet<>();
            for (Long id : calloutIds) {
                if (id == null || validId().apply(id) != null) {
                    continue;
                }
                if (!accessibleCallouts.contains(id)) {
                    defects.putIfAbsent(id, adExtensionNotFound((id)));
                }
                if (!distinctIds.add(id)) {
                    defects.putIfAbsent(id, duplicatedAdExtensionId(id));
                }
            }
            defects.values().forEach(lvb.getResult()::addError);

            return lvb.getResult();
        };
    }

    private Set<Long> getAccessibleCallouts(int shard, ClientId clientId, Collection<BannerWithCallouts> banners) {
        Set<Long> calloutIds = StreamEx.of(banners)
                .map(BannerWithCallouts::getCalloutIds)
                .nonNull()
                .flatMap(List::stream)
                .toSet();

        return calloutRepository.getExistingCalloutIds(shard,
                clientId,
                calloutIds,
                Boolean.FALSE);
    }

}
