package ru.yandex.direct.core.entity.banner.type.contentpromo;

import java.util.function.Predicate;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContent;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContentType;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.DefaultValidator;

import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.contentPromotionCollectionIsInaccessible;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.contentPromotionVideoIsInaccessible;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.contentTypeNotMatchesAdGroupContentType;
import static ru.yandex.direct.core.entity.contentpromotion.ContentPromotionTypeConverters.contentPromotionContentTypeToContentPromotionAdgroupType;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

public class ContentPromotionIdValidator implements DefaultValidator<Long> {

    // если null, значит указанная в баннере группа не найдена (может быть только при создании баннера)
    private final ContentPromotionAdgroupType contentPromotionAdgroupType;

    // если null, значит контент по указанному в баннере contentPromotionId не найден
    private final ContentPromotionContent content;

    private ContentPromotionIdValidator(ContentPromotionAdgroupType contentPromotionAdgroupType,
                                        ContentPromotionContent content) {
        this.contentPromotionAdgroupType = contentPromotionAdgroupType;
        this.content = content;
    }

    public static ContentPromotionIdValidator contentPromotionIdValidator(
            @Nullable ContentPromotionAdgroupType contentPromotionAdgroupType,
            @Nullable ContentPromotionContent content) {
        return new ContentPromotionIdValidator(contentPromotionAdgroupType, content);
    }

    @Override
    public ValidationResult<Long, Defect> apply(Long contentPromotionId) {
        return ItemValidationBuilder.<Long, Defect>of(contentPromotionId)
                .check(notNull())
                .check(contentExists())
                .check(contentIsAccessible())
                .check(contentTypeMatchesAdGroupContentType())
                .getResult();
    }

    private Constraint<Long, Defect> contentExists() {
        return Constraint.fromPredicate(id -> content != null, objectNotFound());
    }

    private Constraint<Long, Defect> contentIsAccessible() {
        Predicate<Long> predicate = id -> content == null || !content.getIsInaccessible();
        return Constraint.fromPredicateAndDefectSupplier(predicate,
                () -> contentPromotionIsInaccessible(content.getType()));
    }

    private Constraint<Long, Defect> contentTypeMatchesAdGroupContentType() {
        Predicate<Long> predicate = id -> {
            if (content == null || contentPromotionAdgroupType == null) {
                return true;
            }
            ContentPromotionAdgroupType correspondingAdGroupContentTypeForCurrentContent =
                    contentPromotionContentTypeToContentPromotionAdgroupType(content.getType());
            return contentPromotionAdgroupType == correspondingAdGroupContentTypeForCurrentContent;
        };
        return Constraint.fromPredicate(predicate, contentTypeNotMatchesAdGroupContentType());
    }

    private Defect<Void> contentPromotionIsInaccessible(ContentPromotionContentType contentType) {
        switch (contentType) {
            case VIDEO:
                return contentPromotionVideoIsInaccessible();
            case COLLECTION:
                return contentPromotionCollectionIsInaccessible();
            case SERVICE:
            case EDA:
                return null;    // Content Promotion Service and Eda cannot be inaccessible
            default:
                throw new IllegalStateException("No such ContentPromotionContentType: " + contentType);
        }
    }
}
