package ru.yandex.direct.core.entity.banner.type.creative;

import java.util.Map;

import ru.yandex.direct.core.entity.banner.model.BannerWithCreative;
import ru.yandex.direct.core.entity.banner.model.CpmBanner;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelChangesValidationBuilder;

import static ru.yandex.direct.core.entity.banner.type.creative.BannerWithCreativeConstraints.creativeIsAdaptiveHasNotChanged;
import static ru.yandex.direct.core.entity.banner.type.creative.BannerWithCreativeConstraints.creativeSizeIsNotChanged;

/**
 * Валидация изменений креатива
 */
public class BannerCreativeModificationValidator implements Validator<ModelChanges<BannerWithCreative>, Defect> {

    private final Map<Long, Creative> clientCreativesByIds;
    private final Map<Long, Long> oldCreativeIdByBannerId;
    private final Class<?> runtimeClass;

    /**
     * @param clientCreativesByIds    - содержит старые и новые креативы, которые есть в БД
     * @param oldCreativeIdByBannerId - содержит только те креативы, которые были в баннерах до применения изменений
     */
    public BannerCreativeModificationValidator(Map<Long, Creative> clientCreativesByIds,
                                                  Map<Long, Long> oldCreativeIdByBannerId,
                                                  Class<?> runtimeClass) {
        this.clientCreativesByIds = clientCreativesByIds;
        this.oldCreativeIdByBannerId = oldCreativeIdByBannerId;
        this.runtimeClass = runtimeClass;
    }

    @Override
    public ValidationResult<ModelChanges<BannerWithCreative>, Defect> apply(ModelChanges<BannerWithCreative> changes) {


        boolean needCheckSize = computeNeedCheckSize(changes);

        ModelChangesValidationBuilder<BannerWithCreative> vb = ModelChangesValidationBuilder.of(changes);

        vb
                .check(creativeSizeIsNotChanged(clientCreativesByIds::get, oldCreativeIdByBannerId::get),
                        When.isValidAnd(When.isTrue(needCheckSize)))
                .check(creativeIsAdaptiveHasNotChanged(clientCreativesByIds::get, oldCreativeIdByBannerId::get),
                        When.isValid());

        return vb.getResult();
    }

    private boolean computeNeedCheckSize(ModelChanges<BannerWithCreative> changes) {
        if (runtimeClass.isAssignableFrom(CpmBanner.class)) {
            Long bannerId = changes.getId();
            Long oldCreativeId = oldCreativeIdByBannerId.get(bannerId);
            Creative oldCreative = clientCreativesByIds.get(oldCreativeId);
            return oldCreative != null
                    && oldCreative.getType() != CreativeType.CPM_VIDEO_CREATIVE
                    && oldCreative.getType() != CreativeType.BANNERSTORAGE
                    && oldCreative.getType() != CreativeType.CPM_OVERLAY;
        }
        return true;
    }
}
