package ru.yandex.direct.core.entity.banner.type.creative;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.BannerWithCreativeModeration;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.regions.GeoTree;

import static java.util.Collections.emptyMap;
import static java.util.function.Function.identity;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.PERF_CREATIVES;
import static ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class BannerCreativeRepository {

    private final DslContextProvider ppcDslContextProvider;
    private final BannerTypedRepository bannerTypedRepository;

    @Autowired
    public BannerCreativeRepository(DslContextProvider ppcDslContextProvider,
                                       BannerTypedRepository bannerTypedRepository) {
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.bannerTypedRepository = bannerTypedRepository;
    }

    public Map<Long, BannerWithCreativeModeration> getBannersWithNotNullCreativeByBannerId(
            int shard, Collection<Long> bannerIds) {
        List<BannerWithCreativeModeration> bannerCreatives = bannerTypedRepository.getSafely(shard, bannerIds,
                BannerWithCreativeModeration.class);
        return StreamEx.of(bannerCreatives)
                .filter(banner -> banner.getCreativeId() != null)
                .toMap(BannerWithCreativeModeration::getId, identity());
    }

    /**
     * Получает маппинг CREATIVE_ID по списку ID баннеров
     */
    public Map<Long, Long> getBannerIdToCreativeId(int shard, Collection<Long> bannerIds) {
        return ppcDslContextProvider.ppc(shard).select(BANNERS_PERFORMANCE.BID, BANNERS_PERFORMANCE.CREATIVE_ID)
                .from(BANNERS_PERFORMANCE)
                .where(BANNERS_PERFORMANCE.BID.in(bannerIds))
                .fetchMap(BANNERS_PERFORMANCE.BID, BANNERS_PERFORMANCE.CREATIVE_ID);
    }

    /**
     * Получает список ID креативов по списку ID баннеров
     */
    public List<Long> getBannerPerformanceCreativeIds(DSLContext context, List<Long> bannerIds) {
        return context
                .select(BANNERS_PERFORMANCE.BANNER_CREATIVE_ID)
                .from(BANNERS_PERFORMANCE)
                .where(BANNERS_PERFORMANCE.BID.in(bannerIds))
                .fetch(BANNERS_PERFORMANCE.BANNER_CREATIVE_ID);
    }

    /**
     * Получает список ID баннеров по списку ID креативов
     */
    public List<Long> getBannerPerformanceBannerIds(int shard, List<Long> creativeIds) {
        return getBannerPerformanceBannerIds(ppcDslContextProvider.ppc(shard), creativeIds);
    }

    public List<Long> getBannerPerformanceBannerIds(DSLContext context, List<Long> creativeIds) {
        return context
                .selectDistinct(BANNERS_PERFORMANCE.BID)
                .from(BANNERS_PERFORMANCE)
                .where(BANNERS_PERFORMANCE.CREATIVE_ID.in(creativeIds))
                .fetch(BANNERS_PERFORMANCE.BID);
    }

    public Map<Long, String> getJoinedGeo(int shard, GeoTree geoTree, Collection<Long> creativeIds) {
        return getJoinedGeo(ppcDslContextProvider.ppc(shard).configuration(), geoTree, creativeIds);
    }

    /**
     * На вход принимает список пар id креативов
     * <p>
     * Возвращает объединенный по всем группам, которым принадлежит креатив, список георегионов
     * Результат:
     * {
     * id_креатива1 => 'id_региона1,id_региона2,id_региона3 ...',
     * id_креатива2 => 'id_региона4,id_региона3,id_региона7 ...',
     * ...
     * }
     */
    public Map<Long, String> getJoinedGeo(Configuration config, GeoTree geoTree, Collection<Long> creativeIds) {
        if (creativeIds.isEmpty()) {
            return emptyMap();
        }

        Map<Long, List<String>> geoFromDb = config.dsl()
                .select(BANNERS_PERFORMANCE.CREATIVE_ID, PHRASES.GEO)
                .from(BANNERS_PERFORMANCE)
                .join(PHRASES).on(BANNERS_PERFORMANCE.PID.eq(PHRASES.PID))
                .where(BANNERS_PERFORMANCE.CREATIVE_ID.in(creativeIds))
                .fetchGroups(BANNERS_PERFORMANCE.CREATIVE_ID, PHRASES.GEO);

        Map<Long, String> result = new HashMap<>();
        StreamEx.of(creativeIds)
                .mapToEntry(creativeId -> nvl(geoFromDb.get(creativeId), Collections.<String>emptyList()))
                .mapValues(geo -> StreamEx.of(geo)
                        .map(g -> g != null ? g.split(",") : new String[0])
                        .flatMap(Arrays::stream)
                        .map(Long::valueOf)
                        .toSet())
                .mapValues(geo -> mapList(geoTree.getModerationCountries(geo), Object::toString))
                .mapValues(geo -> StringUtils.trimToNull(String.join(",", geo)))
                .forKeyValue(result::put);
        return result;
    }

    /**
     * Фильтрует баннеры, оставляет только с креативом расхлопа
     */
    public List<Long> filterExpandedBanner(int shard, List<Long> bannerIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(BANNERS_PERFORMANCE.BID)
                .from(BANNERS_PERFORMANCE)
                .innerJoin(PERF_CREATIVES).on(BANNERS_PERFORMANCE.CREATIVE_ID.eq(PERF_CREATIVES.CREATIVE_ID))
                .where(BANNERS_PERFORMANCE.BID.in(bannerIds))
                .and(PERF_CREATIVES.EXPANDED_PREVIEW_URL.isNotNull())
                .fetch(BANNERS_PERFORMANCE.BID);
    }
}
