package ru.yandex.direct.core.entity.banner.type.creative;

import java.util.List;
import java.util.Set;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.container.BannerAdditionalActionsContainer;
import ru.yandex.direct.core.entity.banner.container.BannersAddOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerCreativeStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithCreative;
import ru.yandex.direct.core.entity.banner.model.PerformanceBanner;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.service.BannerModerationUtils;
import ru.yandex.direct.core.entity.banner.service.type.add.AbstractBannerAddOperationTypeSupport;
import ru.yandex.direct.core.entity.creative.service.CreativeService;

import static ru.yandex.direct.core.entity.banner.service.BannerModerationUtils.isEffectiveSaveDraft;
import static ru.yandex.direct.core.entity.banner.service.validation.type.BannerTypeValidationPredicates.isCpmBanner;
import static ru.yandex.direct.core.entity.banner.type.creative.BannerWithCreativeUtils.isBannerStorage;
import static ru.yandex.direct.core.entity.banner.type.creative.BannerWithCreativeUtils.isOverlayBanner;
import static ru.yandex.direct.core.entity.banner.type.creative.BannerWithCreativeUtils.moderateBannerStorageCreative;
import static ru.yandex.direct.utils.FunctionalUtils.selectList;

@Component
public class BannerWithCreativeAddOperationTypeSupport
        extends AbstractBannerAddOperationTypeSupport<BannerWithCreative> {

    private final CreativeService creativeService;

    @Autowired
    public BannerWithCreativeAddOperationTypeSupport(CreativeService creativeService) {
        this.creativeService = creativeService;
    }

    @Override
    public Class<BannerWithCreative> getTypeClass() {
        return BannerWithCreative.class;
    }

    @Override
    public void onPreValidated(BannersAddOperationContainer addContainer, List<BannerWithCreative> models) {
        synchronizeTextBannersVideoAdditionCreatives(addContainer, models);
    }

    private void synchronizeTextBannersVideoAdditionCreatives(BannersAddOperationContainer addContainer,
                                                              List<BannerWithCreative> models) {
        List<TextBanner> textBanners = selectList(models, TextBanner.class);

        Set<Long> textBannersCreativeIds = StreamEx.of(textBanners)
                .map(TextBanner::getCreativeId)
                .nonNull()
                .toSet();

        if (!textBannersCreativeIds.isEmpty()) {
            creativeService.synchronizeVideoAdditionCreatives(addContainer.getShard(), addContainer.getClientId(),
                    textBannersCreativeIds);
        }
    }

    @Override
    public void beforeExecution(BannersAddOperationContainer addContainer, List<BannerWithCreative> models) {
        models.stream()
                .filter(b -> b.getCreativeId() != null)
                .forEach(banner -> banner.setCreativeStatusModerate(creativeStatusModerate(addContainer, banner)));
    }

    private static BannerCreativeStatusModerate creativeStatusModerate(BannersAddOperationContainer container,
                                                                       BannerWithCreative banner) {
        var adGroup = container.getAdGroup(banner);
        if (adGroup.getType() == AdGroupType.CPM_GEOPRODUCT && container.isCpmGeoProductAutoModeration()) {
            return BannerCreativeStatusModerate.YES;
        }

        if (adGroup.getType() == AdGroupType.CPM_GEO_PIN && container.isCpmGeoPinAutoModeration()) {
            return BannerCreativeStatusModerate.YES;
        }

        // Оверлейные баннеры не отправляем на модерацию
        if (isOverlayBanner(banner, container.getCreativeByIdMap())) {
            return BannerCreativeStatusModerate.YES;
        }

        if (banner instanceof PerformanceBanner) {
            if (BannerModerationUtils.isEffectiveSaveDraft(container, banner)) {
                return BannerCreativeStatusModerate.NEW;
            } else {
                return BannerCreativeStatusModerate.YES;
            }
        }

        var effectiveSaveDraft = isEffectiveSaveDraft(container, banner);
        return effectiveSaveDraft ? BannerCreativeStatusModerate.NEW : BannerCreativeStatusModerate.READY;
    }

    @Override
    public void addToAdditionalActionsContainer(BannerAdditionalActionsContainer additionalActionsContainer,
                                                BannersAddOperationContainer addModelContainer,
                                                List<BannerWithCreative> models) {
        List<PerformanceBanner> performanceBanners = selectList(models, PerformanceBanner.class);
        updateAdGroupsForPerformanceBanners(additionalActionsContainer, addModelContainer, performanceBanners);
        updateCreativesForPerformanceBanners(additionalActionsContainer, addModelContainer, performanceBanners);

        updateCreativesForBannerStorage(additionalActionsContainer, addModelContainer, models);
    }

    /**
     * специфическая модерация CpmBanner с креативом из BannerStorage
     */
    private void updateCreativesForBannerStorage(BannerAdditionalActionsContainer additionalActionsContainer,
                                                 BannersAddOperationContainer addModelContainer,
                                                 List<BannerWithCreative> models) {
        models.forEach(banner -> {
            if (isCpmBanner(banner) && isBannerStorage(banner, addModelContainer.getCreativeByIdMap())) {
                moderateBannerStorageCreative(additionalActionsContainer, banner, addModelContainer);
            }
        });
    }

    private void updateAdGroupsForPerformanceBanners(BannerAdditionalActionsContainer additionalActionsContainer,
                                                     BannersAddOperationContainer addModelContainer,
                                                     List<PerformanceBanner> models) {
        Set<Long> adGroupIds = StreamEx.of(models)
                .filter(banner -> !BannerModerationUtils.isEffectiveSaveDraft(addModelContainer, banner))
                .map(PerformanceBanner::getAdGroupId)
                .nonNull()
                .toSet();
        additionalActionsContainer.addAdGroupsIdsForBSResync(adGroupIds);
        additionalActionsContainer.addDraftPerformanceAdGroupsAsModerated(adGroupIds);
    }

    private void updateCreativesForPerformanceBanners(BannerAdditionalActionsContainer additionalActionsContainer,
                                                      BannersAddOperationContainer addModelContainer,
                                                      List<PerformanceBanner> models) {
        Set<Long> creativeIds = StreamEx.of(models)
                .map(BannerWithCreative::getCreativeId)
                .nonNull()
                .toSet();

        additionalActionsContainer.addCreativeIdsToSetGeo(creativeIds);

        Set<Long> performanceCreativeIdsToModerate = StreamEx.of(models)
                .filter(banner -> !BannerModerationUtils.isEffectiveSaveDraft(addModelContainer, banner))
                .map(BannerWithCreative::getCreativeId)
                .nonNull()
                .toSet();

        additionalActionsContainer.addCreativeIdsToModerate(performanceCreativeIdsToModerate);
    }

}
