package ru.yandex.direct.core.entity.banner.type.creative;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.function.LongFunction;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.banner.model.BannerWithCreative;
import ru.yandex.direct.core.entity.banner.type.creative.model.CreativeSizeWithExpand;
import ru.yandex.direct.core.entity.creative.constants.AdminRejectionReason;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.CreativeBusinessType;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.creative.model.StatusModerate;
import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static com.google.common.base.Preconditions.checkState;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.creativeNotFound;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.imageSizeModification;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.inconsistentCreativeAndCpmVideoAdGroupSkippability;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.inconsistentCreativeBusinessTypeToFeedBusinessType;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.inconsistentCreativeFormat;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.inconsistentCreativeGeoToAdGroupGeo;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.inconsistentCreativeTypeToBannerType;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.priceSalesDisallowedCreativeTemplate;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.requiredCreativesWithCanvasOrHtml5Types;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.requiredCreativesWithCanvasTypeOnly;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.requiredCreativesWithCpmAudioTypeOnly;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.requiredCreativesWithCpmVideoTypeOnly;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.requiredCreativesWithHtml5TypeOnly;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.requiredCreativesWithVideoAdditionTypeOnly;
import static ru.yandex.direct.core.entity.banner.type.creative.BannerWithCreativeConstants.FEEDS_CREATIVES_COMPATIBILITY;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.NON_SKIPPABLE_CPM_VIDEO_LAYOUT_ID;
import static ru.yandex.direct.core.entity.creative.service.add.validation.CreativeDefects.creativeIsAdminRejected;
import static ru.yandex.direct.utils.CommonUtils.ifNotNullOrDefault;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@ParametersAreNonnullByDefault
public class BannerWithCreativeConstraints {

    private BannerWithCreativeConstraints() {
    }


    public static Constraint<Long, Defect> isClientHasCreative(Set<Long> clientCreativeIds) {
        return fromPredicate(clientCreativeIds::contains, creativeNotFound());
    }

    /**
     * Проверяет валидность принадлежности видеокреатива к типу баннера.
     */
    public static Constraint<Long, Defect> isCreativeTypeCorrespondTo(Set<CreativeType> allowedCreativeTypes,
                                                                      Map<Long, Creative> accessibleCreativesByIds) {

        return fromPredicate(creativeId -> {
                    Creative creative = accessibleCreativesByIds.get(creativeId);
                    if (creative == null) {
                        return true;
                    }

                    return allowedCreativeTypes.contains(creative.getType());
                },
                inconsistentCreativeTypeToBannerType());
    }

    public static Constraint<Long, Defect> creativeTypeIsVideoAddition(Map<Long, Creative> accessibleCreativesByIds) {
        return fromPredicate(creativeId -> !accessibleCreativesByIds.containsKey(creativeId) ||
                        accessibleCreativesByIds.get(creativeId).getType() == CreativeType.VIDEO_ADDITION_CREATIVE,
                requiredCreativesWithVideoAdditionTypeOnly());
    }

    public static Constraint<Long, Defect> creativeTypeIsCanvasOrHtml5(Map<Long, Creative> accessibleCreativesByIds) {
        return fromPredicate(creativeId -> !accessibleCreativesByIds.containsKey(creativeId) ||
                        ImmutableSet.of(CreativeType.CANVAS, CreativeType.HTML5_CREATIVE)
                                .contains(accessibleCreativesByIds.get(creativeId).getType()),
                requiredCreativesWithCanvasOrHtml5Types());
    }

    public static Constraint<Long, Defect> creativeTypeIsCanvas(Map<Long, Creative> accessibleCreativesByIds) {
        return fromPredicate(creativeId -> !accessibleCreativesByIds.containsKey(creativeId) ||
                        ImmutableSet.of(CreativeType.CANVAS)
                                .contains(accessibleCreativesByIds.get(creativeId).getType()),
                requiredCreativesWithCanvasTypeOnly());
    }

    public static Constraint<Long, Defect> creativeTypeIsCpmAudioCreative(
            Map<Long, Creative> accessibleCreativesByIds) {
        return fromPredicate(creativeId -> !accessibleCreativesByIds.containsKey(creativeId) ||
                        accessibleCreativesByIds.get(creativeId).getType() == CreativeType.BANNERSTORAGE ||
                        accessibleCreativesByIds.get(creativeId).getType() == CreativeType.CPM_AUDIO_CREATIVE,
                requiredCreativesWithCpmAudioTypeOnly());
    }

    public static Constraint<Long, Defect> creativeTypeIsCpmVideoCreative(
            Map<Long, Creative> accessibleCreativesByIds) {
        return fromPredicate(creativeId -> !accessibleCreativesByIds.containsKey(creativeId) ||
                        accessibleCreativesByIds.get(creativeId).getType() == CreativeType.CPM_VIDEO_CREATIVE ||
                        accessibleCreativesByIds.get(creativeId).getType() == CreativeType.BANNERSTORAGE ||
                        accessibleCreativesByIds.get(creativeId).getType() == CreativeType.CPM_OVERLAY,
                requiredCreativesWithCpmVideoTypeOnly());
    }

    public static Constraint<Long, Defect> creativeMatchesAdGroupSkippability(
            Map<Long, Creative> accessibleCreativesByIds, boolean isNonSkippableCpmVideoAdGroup) {
        return fromPredicate(creativeId ->
                        (ifNotNullOrDefault(accessibleCreativesByIds.get(creativeId).getLayoutId(),
                                NON_SKIPPABLE_CPM_VIDEO_LAYOUT_ID::contains, false))
                                == isNonSkippableCpmVideoAdGroup,
                inconsistentCreativeAndCpmVideoAdGroupSkippability());
    }

    /**
     * Проверка креатива, что его layoutId разрешен пакетом
     *
     * @param creativesByIds           мапа доступных пользователю креативов по их идентификатору
     * @param allowedCreativeTemplates мапа разрешенных layoutId, разложенных по типам креатива
     */
    public static Constraint<Long, Defect> isAllowedCreativeTemplates(
            Map<Long, Creative> creativesByIds,
            Map<CreativeType, List<Long>> allowedCreativeTemplates) {
        return fromPredicate(creativeId -> {
                    Creative creative = creativesByIds.get(creativeId);
                    return allowedCreativeTemplates.containsKey(creative.getType()) &&
                            allowedCreativeTemplates.get(creative.getType()) != null &&
                            allowedCreativeTemplates.get(creative.getType()).contains(creativeTemplateId(creative));
                },
                priceSalesDisallowedCreativeTemplate());
    }

    /**
     * Возвращает номер шаблона креатива.
     * Его называют шаблоном, пресетом или layout.
     */
    public static Long creativeTemplateId(Creative creative) {
        var type = creative.getType();
        if (type != null && type == CreativeType.BANNERSTORAGE) {
            return creative.getTemplateId();
        }
        return creative.getLayoutId();
    }

    /**
     * Проверка идентификатора на то, что соответствующий ему креатив типа Html5
     *
     * @param accessibleCreativesByIds мапа доступных пользователю креативов по их идентификатору
     */
    public static Constraint<Long, Defect> creativeTypeIsHtml5(Map<Long, Creative> accessibleCreativesByIds) {
        return fromPredicate(creativeId -> !accessibleCreativesByIds.containsKey(creativeId) ||
                        CreativeType.HTML5_CREATIVE.equals(accessibleCreativesByIds.get(creativeId).getType()),
                requiredCreativesWithHtml5TypeOnly());
    }

    public static Constraint<Long, Defect> creativeIsNotAdminRejected(Map<Long, Creative> accessibleCreativesByIds) {
        return creativeId -> {
            Creative creative = accessibleCreativesByIds.get(creativeId);
            return creative == null || creative.getStatusModerate() != StatusModerate.ADMINREJECT
                    ? null
                    : creativeIsAdminRejected(
                    AdminRejectionReason.parse(creative.getModerationInfo().getAdminRejectReason()));
        };
    }

    public static Constraint<Long, Defect> isCreativeBusinessTypeCorrespondTo(BusinessType feedBusinessType,
                                                                              Map<Long, Creative> accessibleCreativesByIds) {
        return fromPredicate(creativeId -> {
                    Creative creative = accessibleCreativesByIds.get(creativeId);
                    if (creative == null) {
                        return true;
                    }

                    Set<CreativeBusinessType> types = FEEDS_CREATIVES_COMPATIBILITY.get(feedBusinessType);
                    checkState(types != null, "allowed creatives business_type must defined in map");

                    return types.contains(creative.getBusinessType());
                },
                inconsistentCreativeBusinessTypeToFeedBusinessType());
    }

    public static <B extends BannerWithCreative> Constraint<ModelChanges<B>, Defect> creativeSizeIsNotChanged(
            Function<Long, Creative> creativeById, Function<Long, Long> creativeIdByBannerId) {
        return fromPredicate(mc -> {
            if (!mc.isPropChanged(B.CREATIVE_ID)) {
                return true;
            }
            Creative oldCreative = creativeById.apply(creativeIdByBannerId.apply(mc.getId()));
            Creative newCreative = creativeById.apply(mc.getChangedProp(B.CREATIVE_ID));
            if (oldCreative == null || newCreative == null) {
                return true;
            }
            return Long.compare(oldCreative.getWidth(), nvl(newCreative.getWidth(), oldCreative.getWidth())) == 0 &&
                    Long.compare(oldCreative.getHeight(), nvl(newCreative.getHeight(), oldCreative.getHeight())) == 0;
        }, imageSizeModification());
    }

    /**
     * Проверка того, что адаптивный баннер не поменяли на неадаптивный и наоборот.
     */
    public static <B extends BannerWithCreative> Constraint<ModelChanges<B>, Defect> creativeIsAdaptiveHasNotChanged(
            LongFunction<Creative> creativeById,
            LongFunction<Long> creativeIdByBannerId
    ) {
        return fromPredicate(mc -> {
            if (!mc.isPropChanged(B.CREATIVE_ID)) {
                return true;
            }

            var oldCreativeId = creativeIdByBannerId.apply(mc.getId());
            var newCreativeId = mc.getPropIfChanged(B.CREATIVE_ID);

            if (oldCreativeId == null || newCreativeId == null) {
                return true;
            }

            var oldCreative = creativeById.apply(oldCreativeId);
            var newCreative = creativeById.apply(newCreativeId);

            return oldCreative.getIsAdaptive().equals(newCreative.getIsAdaptive());
        }, imageSizeModification());
    }

    public static Constraint<Long, Defect> checkCreativeFormat(
            Map<Long, Creative> accessibleCreativesByIds,
            Set<Set<CreativeSizeWithExpand>> allowedCreativeSizesWithExpand) {

        return fromPredicate(creativeId -> {
                    Creative creative = accessibleCreativesByIds.get(creativeId);
                    if (creative == null) {
                        return true;
                    }
                    CreativeSizeWithExpand creativeSizeWithExpand = new CreativeSizeWithExpand(creative);
                    return allowedCreativeSizesWithExpand.stream().anyMatch(e -> e.contains(creativeSizeWithExpand));
                },
                inconsistentCreativeFormat());
    }

    private static boolean isConsistentCreativeGeoToAdGroupCountries(@Nullable Creative creative,
                                                                     Collection<Long> adGroupCountries) {
        if ((creative == null)
                || (creative.getSumGeo() == null)
                || (creative.getStatusModerate() == StatusModerate.NEW)
                || (creative.getStatusModerate() == StatusModerate.ERROR)
                || isEmpty(adGroupCountries)) {
            return true;
        }
        return creative.getSumGeo().containsAll(adGroupCountries);
    }

    public static Constraint<Long, Defect> isCreativeSumGeoCorrespondTo(Collection<Long> adGroupCountries,
                                                                        Map<Long, Creative> accessibleCreativesByIds) {
        return fromPredicate(creativeId -> isConsistentCreativeGeoToAdGroupCountries(
                accessibleCreativesByIds.get(creativeId),
                adGroupCountries),
                inconsistentCreativeGeoToAdGroupGeo());
    }

    public static boolean isConsistentCreativeGeoToAdGroupGeo(Collection<Creative> adGroupCreatives,
                                                              Set<Long> adGroupCountries) {
        return StreamEx.of(adGroupCreatives)
                .allMatch(creative -> isConsistentCreativeGeoToAdGroupCountries(creative, adGroupCountries));
    }
}
