package ru.yandex.direct.core.entity.banner.type.creative;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.BannerWithCreative;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.creative.service.CreativeService;
import ru.yandex.direct.core.entity.creative.service.CreativeUtils;
import ru.yandex.direct.dbutil.model.ClientId;

import static java.util.Collections.emptyMap;
import static ru.yandex.direct.core.entity.creative.service.CreativeUtils.extractModerationInfoPropertyFromCreative;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Component
public class BannerWithCreativeHelper {
    private final CreativeRepository creativeRepository;
    private final CreativeService creativeService;

    @Autowired
    public BannerWithCreativeHelper(CreativeRepository creativeRepository,
                                       CreativeService creativeService) {
        this.creativeRepository = creativeRepository;
        this.creativeService = creativeService;
    }

    public Map<Long, Creative> getBannerCreatives(
            ClientId clientId,
            List<BannerWithCreative> banners) {
        Set<Long> creativeIds = StreamEx.of(banners)
                .map(BannerWithCreative::getCreativeId)
                .filter(Objects::nonNull)
                .toSet();
        if (creativeIds.isEmpty()) {
            return emptyMap();
        }
        List<Creative> creatives = creativeService.get(clientId, creativeIds,
                List.of(CreativeType.CPM_OVERLAY, CreativeType.BANNERSTORAGE, CreativeType.CANVAS));
        return listToMap(creatives, Creative::getId);
    }

    public Map<Long, String> getCreativesModerationTitles(
            int shard,
            Collection<? extends BannerWithCreative> bannersWithCreatives) {
        return getCreativesModerationProperties(shard, bannersWithCreatives, CreativeUtils.MODERATION_PROPERTY_TITLE);
    }

    public Map<Long, String> getCreativesModerationBodies(
            int shard,
            Collection<? extends BannerWithCreative> bannersWithCreatives) {
        return getCreativesModerationProperties(shard, bannersWithCreatives, CreativeUtils.MODERATION_PROPERTY_BODY);
    }

    public Map<Long, String> getCreativesModerationProperties(
            int shard,
            Collection<? extends BannerWithCreative> bannersWithCreatives,
            String property) {
        Set<Long> creativeIds = StreamEx.of(bannersWithCreatives)
                .map(BannerWithCreative::getCreativeId)
                .nonNull()
                .toSet();

        List<Creative> creatives = creativeRepository.getCreatives(shard, creativeIds);

        return StreamEx.of(creatives)
                .mapToEntry(creative -> extractModerationInfoPropertyFromCreative(creative, property))
                .mapKeys(Creative::getId)
                .filterValues(Objects::nonNull)
                .toMap();
    }
}
