package ru.yandex.direct.core.entity.banner.type.creative;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jetbrains.annotations.NotNull;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.container.BannersOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerCreativeStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithCreative;
import ru.yandex.direct.core.entity.banner.repository.type.AbstractFlatRelatedEntityRepositoryTypeSupport;
import ru.yandex.direct.core.entity.banner.repository.type.ModifiedPaths;
import ru.yandex.direct.core.entity.banner.repository.type.ModifiedPathsBuilder;
import ru.yandex.direct.dbschema.ppc.tables.records.BannersPerformanceRecord;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.grut.objects.proto.client.Schema;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;

@Component
@ParametersAreNonnullByDefault
public class BannerWithCreativeRepositoryTypeSupport
        extends AbstractFlatRelatedEntityRepositoryTypeSupport
        <BannerWithCreative, BannersPerformanceRecord> {

    private static final Set<ModelProperty<? super BannerWithCreative, ?>> GRUT_PROPERTIES =
            Set.of(BannerWithCreative.CREATIVE_ID);

    private final ShardHelper shardHelper;

    @Autowired
    public BannerWithCreativeRepositoryTypeSupport(DslContextProvider dslContextProvider,
                                                   ShardHelper shardHelper) {
        super(dslContextProvider, BANNERS_PERFORMANCE.BID, createMapper());
        this.shardHelper = shardHelper;
    }

    @Override
    protected void addEntities(DSLContext context,
                               BannerRepositoryContainer updateParameters,
                               Collection<BannerWithCreative> entities) {
        List<Long> bannerCreativeIds = shardHelper.generateBannerCreativeIds(entities.size());
        var helper = new InsertHelper<>(context, BANNERS_PERFORMANCE);
        StreamEx.of(bannerCreativeIds).zipWith(entities.stream())
                .forKeyValue((id, bannerCreative) -> {
                    helper.set(BANNERS_PERFORMANCE.BANNER_CREATIVE_ID, id);
                    helper.add(getJooqMapper(), bannerCreative)
                            .newRecord();
                });
        helper.executeIfRecordsAdded();
    }

    @Override
    protected boolean isAddEntity(BannerWithCreative model) {
        return model.getCreativeId() != null;
    }

    @Override
    protected boolean isAddEntity(AppliedChanges<BannerWithCreative> appliedChange) {
        return appliedChange.assigned(BannerWithCreative.CREATIVE_ID);
    }

    @Override
    protected boolean isUpdateEntity(AppliedChanges<BannerWithCreative> appliedChange) {
        return (appliedChange.changed(BannerWithCreative.CREATIVE_ID)
                || appliedChange.changed(BannerWithCreative.CREATIVE_STATUS_MODERATE)
                || appliedChange.changed(BannerWithCreative.SHOW_TITLE_AND_BODY))
                && appliedChange.getOldValue(BannerWithCreative.CREATIVE_ID) != null
                && appliedChange.getNewValue(BannerWithCreative.CREATIVE_ID) != null;
    }

    @Override
    protected boolean isDeleteEntity(AppliedChanges<BannerWithCreative> appliedChange) {
        return appliedChange.deleted(BannerWithCreative.CREATIVE_ID);
    }

    @Override
    public Class<BannerWithCreative> getTypeClass() {
        return BannerWithCreative.class;
    }

    private static JooqMapper<BannerWithCreative> createMapper() {
        return JooqMapperBuilder.<BannerWithCreative>builder()
                .map(property(BannerWithCreative.CREATIVE_ID, BANNERS_PERFORMANCE.CREATIVE_ID))
                .map(convertibleProperty(BannerWithCreative.CREATIVE_STATUS_MODERATE,
                        BANNERS_PERFORMANCE.STATUS_MODERATE,
                        BannerCreativeStatusModerate::fromSource,
                        BannerCreativeStatusModerate::toSource))
                .map(convertibleProperty(
                        BannerWithCreative.SHOW_TITLE_AND_BODY,
                        BANNERS_PERFORMANCE.SHOW_TITLE_AND_BODY,
                        RepositoryUtils::booleanFromLong,
                        RepositoryUtils::nullSafeBooleanToLong
                ))
                .readProperty(BannerWithCreative.CREATIVE_RELATION_ID,
                        fromField(BANNERS_PERFORMANCE.BANNER_CREATIVE_ID))
                .writeField(BANNERS_PERFORMANCE.BID, fromProperty(BannerWithCreative.ID))
                .writeField(BANNERS_PERFORMANCE.PID, fromProperty(BannerWithCreative.AD_GROUP_ID))
                .writeField(BANNERS_PERFORMANCE.CID, fromProperty(BannerWithCreative.CAMPAIGN_ID))
                .build();
    }

    @Override
    public Set<ModelProperty<? super BannerWithCreative, ?>> getGrutSupportedProperties() {
        return GRUT_PROPERTIES;
    }

    @Override
    public Map<Long, ModifiedPaths> applyToGrutObjects(
            @NotNull Map<Long, Schema.TBannerV2.Builder> bannerBuilders,
            @NotNull Collection<AppliedChanges<BannerWithCreative>> appliedChangesList,
            @NotNull BannersOperationContainer operationContainer) {
        Map<Long, ModifiedPaths> modifiedPathsMap = new HashMap<>();
        for (var appliedChanges : appliedChangesList) {
            if (appliedChanges.getPropertiesForUpdate().contains(BannerWithCreative.CREATIVE_ID)) {
                Long id = appliedChanges.getModel().getId();
                var newValue = appliedChanges.getNewValue(BannerWithCreative.CREATIVE_ID);
                ModifiedPaths modifiedPaths;
                if (newValue != null) {
                    Schema.TBannerV2.Builder bannerBuilder = bannerBuilders.get(id);
                    bannerBuilder.getSpecBuilder().clearCreativeIds();
                    bannerBuilder.getSpecBuilder().addCreativeIds(newValue);

                    modifiedPaths = new ModifiedPathsBuilder()
                            .withSetPaths(Set.of("/spec/creative_ids"))
                            .withCreativeIdToCheckIfExist(newValue)
                            .build();
                } else {
                    modifiedPaths = new ModifiedPaths(emptySet(), Set.of("/spec/creative_ids"));
                }
                modifiedPathsMap.put(id, modifiedPaths);
            }
        }
        return modifiedPathsMap;
    }
}
