package ru.yandex.direct.core.entity.banner.type.creative;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithCreative;
import ru.yandex.direct.core.entity.banner.service.validation.type.update.AbstractBannerUpdateValidationTypeSupport;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Component
public class BannerWithCreativeUpdateValidationTypeSupport
        extends AbstractBannerUpdateValidationTypeSupport<BannerWithCreative> {

    private final BannerWithCreativeValidationContainerFactory creativeValidationContainerFactory;
    private final BannerWithCreativeValidationHelper bannerValidationHelper;
    private final BannerWithCreativeValidatorProvider validatorProvider;

    @Autowired
    public BannerWithCreativeUpdateValidationTypeSupport(
            BannerWithCreativeValidationContainerFactory creativeValidationContainerFactory,
            BannerWithCreativeValidationHelper bannerValidationHelper,
            BannerWithCreativeValidatorProvider validatorProvider) {

        this.creativeValidationContainerFactory = creativeValidationContainerFactory;
        this.bannerValidationHelper = bannerValidationHelper;
        this.validatorProvider = validatorProvider;
    }

    @Override
    public Class<BannerWithCreative> getTypeClass() {
        return BannerWithCreative.class;
    }

    @Override
    public ValidationResult<List<ModelChanges<BannerWithCreative>>, Defect> validateBeforeApply(
            BannersUpdateOperationContainer container,
            ValidationResult<List<ModelChanges<BannerWithCreative>>, Defect> vr,
            Map<Long, BannerWithCreative> unmodifiedModels) {

        Map<Long, Creative> clientCreativesByIds = getClientCreativesByIds(container, vr, unmodifiedModels);

        Map<Long, Long> oldCreativeIdByBannerId = new HashMap<>();
        unmodifiedModels.forEach((bannerId, banner) -> oldCreativeIdByBannerId.put(bannerId, banner.getCreativeId()));

        return new ListValidationBuilder<>(vr)
                .checkEachBy(
                        validatorProvider.creativeModificationValidator(
                                mc -> {
                                    //noinspection unchecked
                                    return (Class<? extends BannerWithCreative>) container.getRuntimeClass(
                                            mc.getId());
                                },
                                clientCreativesByIds,
                                oldCreativeIdByBannerId))
                .getResult();
    }


    /**
     * Получить мапу creativeId -> Creative, в которой содержатся старые и новые креативы
     */
    private Map<Long, Creative> getClientCreativesByIds(
            BannersUpdateOperationContainer container,
            ValidationResult<List<ModelChanges<BannerWithCreative>>, Defect> vr,
            Map<Long, BannerWithCreative> unmodifiedModels) {

        Set<Long> existingCreativeIds = listToSet(unmodifiedModels.values(), BannerWithCreative::getCreativeId);

        Set<Long> allCreativeIds = StreamEx.of(vr.getValue())
                .map(mc -> mc.getPropIfChanged(BannerWithCreative.CREATIVE_ID))
                .append(existingCreativeIds)
                .nonNull()
                .toSet();

        return bannerValidationHelper
                .getAccessibleCreativesByIds(container.getShard(), container.getClientId(), allCreativeIds);
    }


    @Override
    public ValidationResult<List<BannerWithCreative>, Defect> validate(BannersUpdateOperationContainer validationContainer,
                                                                       ValidationResult<List<BannerWithCreative>, Defect> vr) {


        BannerWithCreativeValidationContainer creativeValidationContainer =
                creativeValidationContainerFactory.create(validationContainer, vr.getValue());

        return new ListValidationBuilder<>(vr)
                .checkEachBy(validatorProvider.creativeValidator(creativeValidationContainer))
                .getResult();
    }

}
