package ru.yandex.direct.core.entity.banner.type.creative;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.CpmVideoAdGroup;
import ru.yandex.direct.core.entity.banner.container.BannersOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithCreative;
import ru.yandex.direct.core.entity.banner.model.CpmBanner;
import ru.yandex.direct.core.entity.banner.model.PerformanceBanner;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.dbutil.model.ClientId;

import static ru.yandex.direct.multitype.typesupport.TypeFilteringUtils.filterModelsOfClass;

@Component
public class BannerWithCreativeValidationContainerFactory {

    public BannerWithCreativeValidationContainerFactory(BannerWithCreativeValidationHelper bannerValidationHelper) {
        this.bannerValidationHelper = bannerValidationHelper;
    }

    private final BannerWithCreativeValidationHelper bannerValidationHelper;


    public BannerWithCreativeValidationContainer create(BannersOperationContainer bannerValidationContainer,
                                                        List<BannerWithCreative> allBanners) {

        disallowPerformanceBannerInComplexOperation(bannerValidationContainer, allBanners);

        int shard = bannerValidationContainer.getShard();
        ClientId clientId = bannerValidationContainer.getClientId();

        var adGroupsForBannerOperation = bannerValidationContainer.getUniqueAdGroups();

        Map<Long, Creative> creativesByIds = bannerValidationHelper
                .getAccessibleCreativesByIds(shard, clientId, allBanners);

        List<PerformanceBanner> performanceBanners = filterModelsOfClass(allBanners, PerformanceBanner.class);

        Map<Long, Set<Long>> performanceAdGroupCountriesByAdGroupId =
                bannerValidationHelper.getAdGroupCountriesByAdGroupId(bannerValidationContainer.getClientRegionId(),
                        adGroupsForBannerOperation);
        List<AdGroup> performanceAdGroups = bannerValidationHelper.getAdGroups(shard, performanceBanners);
        Map<Long, BusinessType> performanceFeedBusinessTypeByAdGroupId =
                bannerValidationHelper.getFeedBusinessTypeByAdGroupId(shard, clientId, performanceAdGroups);
        Map<Long, Set<Long>> performanceDuplicatedCreativeIdsByAdGroupId =
                bannerValidationHelper.getDuplicatedCreativeIdsByAdGroupId(shard, performanceBanners);

        List<CpmBanner> cpmBanners = filterModelsOfClass(allBanners, CpmBanner.class);
        List<AdGroup> cpmAdGroups = bannerValidationHelper.getAdGroups(shard, cpmBanners);

        var videoGroupsFromContainer =
                adGroupsForBannerOperation.stream()
                        .filter(e -> e.getType() == AdGroupType.CPM_VIDEO && e instanceof CpmVideoAdGroup)
                        .map(e -> ((CpmVideoAdGroup) e))
                        .collect(Collectors.toList());

        var allGroups =
                Stream.concat(cpmAdGroups.stream(), videoGroupsFromContainer.stream())
                        .filter(Objects::nonNull)
                        .collect(Collectors.toList());

        Set<Long> nonSkippableCpmVideoAdGroupIds = bannerValidationHelper.getNonSkippableAdGroupIds(allGroups);

        Map<Long, CpmPriceCampaign> cpmPriceCampaigns =
                bannerValidationHelper.getCpmPriceCampaigns(shard, bannerValidationContainer.getCampaigns());

        Map<Long, PricePackage> cpmPricePackages =
                bannerValidationHelper.getPricePackages(shard, bannerValidationContainer.getCampaigns());

        return new BannerWithCreativeValidationContainer(
                bannerValidationContainer,
                creativesByIds,
                performanceAdGroupCountriesByAdGroupId,
                performanceFeedBusinessTypeByAdGroupId,
                performanceDuplicatedCreativeIdsByAdGroupId,
                cpmPriceCampaigns,
                cpmPricePackages,
                bannerValidationContainer.getClientEnabledFeatures(),
                nonSkippableCpmVideoAdGroupIds);

    }

    /**
     * Для NewPerformanceBanner нужно получить список adGroupId из баннера.
     * Но в комплексной операции adGroupId не проставляется в баннере (т.к. его еще нет). Поэтому бросаем Exception
     */
    private void disallowPerformanceBannerInComplexOperation(BannersOperationContainer container,
                                                             List<BannerWithCreative> allBanners) {
        if (container.isPartOfComplexOperation()) {
            allBanners.stream()
                    .filter(x -> x.getClass() == PerformanceBanner.class)
                    .findAny()
                    .ifPresent(x -> {
                        throw new IllegalStateException(
                                String.format("Performance banners (id: %s) is not supported in complex operation",
                                        x.getId()));
                    });
        }
    }
}
