package ru.yandex.direct.core.entity.banner.type.creative;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupForBannerOperation;
import ru.yandex.direct.core.entity.adgroup.model.CpmVideoAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.PerformanceAdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.BannerWithCreative;
import ru.yandex.direct.core.entity.banner.model.PerformanceBanner;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.repository.FeedRepository;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.utils.CommonUtils;

import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.multitype.typesupport.TypeFilteringUtils.filterModelsOfClass;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterToSet;
import static ru.yandex.direct.utils.ListUtils.findDuplicates;

@Component
@ParametersAreNonnullByDefault
public class BannerWithCreativeValidationHelper {

    private final CreativeRepository creativeRepository;
    private final AdGroupRepository adGroupRepository;
    private final FeedRepository feedRepository;
    private final ClientGeoService clientGeoService;
    private final BannerTypedRepository bannerRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final PricePackageRepository pricePackageRepository;

    @Autowired
    public BannerWithCreativeValidationHelper(CreativeRepository creativeRepository,
                                              AdGroupRepository adGroupRepository,
                                              FeedRepository feedRepository,
                                              ClientGeoService clientGeoService,
                                              BannerTypedRepository bannerRepository,
                                              CampaignTypedRepository campaignTypedRepository,
                                              PricePackageRepository pricePackageRepository) {
        this.creativeRepository = creativeRepository;
        this.adGroupRepository = adGroupRepository;
        this.feedRepository = feedRepository;
        this.clientGeoService = clientGeoService;
        this.bannerRepository = bannerRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.pricePackageRepository = pricePackageRepository;
    }

    public Map<Long, Creative> getAccessibleCreativesByIds(int shard, ClientId clientId,
                                                           Collection<BannerWithCreative> banners) {
        var creativeIds = mapAndFilterToSet(banners, BannerWithCreative::getCreativeId, CommonUtils::isValidId);
        return getAccessibleCreativesByIds(shard, clientId, creativeIds);
    }

    public Map<Long, Creative> getAccessibleCreativesByIds(int shard, ClientId clientId,
                                                           Set<Long> creativeIds) {
        Collection<Creative> savedCreatives = creativeRepository.getCreatives(shard, clientId, creativeIds);
        return listToMap(savedCreatives, Creative::getId);
    }

    public Map<Long, BusinessType> getFeedBusinessTypeByAdGroupId(int shard, ClientId clientId, List<AdGroup> groups) {

        List<PerformanceAdGroup> performanceAdGroups = filterModelsOfClass(groups, PerformanceAdGroup.class);

        Set<Long> feedIds = listToSet(performanceAdGroups, PerformanceAdGroup::getFeedId);
        Map<Long, Feed> feedByFeedId = getFeeds(shard, clientId, feedIds);

        return StreamEx.of(performanceAdGroups)
                .mapToEntry(
                        AdGroup::getId,
                        adGroup -> Optional.ofNullable(adGroup.getFeedId())
                                .map(feedByFeedId::get)
                                .map(Feed::getBusinessType)
                                .orElse(null))
                .nonNullValues()
                .toMap();
    }

    public Set<Long> getNonSkippableAdGroupIds(List<AdGroup> groups) {
        List<CpmVideoAdGroup> nonSkippableGroups = filterList(filterModelsOfClass(groups, CpmVideoAdGroup.class),
                e -> e.getIsNonSkippable() != null);
        return listToSet(filterAndMapList(nonSkippableGroups, CpmVideoAdGroup::getIsNonSkippable, ModelWithId::getId));
    }

    public List<AdGroup> getAdGroups(int shard, Collection<? extends BannerWithAdGroupId> banners) {
        Set<Long> adGroupIds = listToSet(banners, BannerWithAdGroupId::getAdGroupId);
        return adGroupRepository.getAdGroups(shard, adGroupIds);
    }

    private Map<Long, Feed> getFeeds(int shard, ClientId clientId, Set<Long> feedIds) {
        List<Feed> feeds = feedRepository.get(shard, clientId, feedIds);
        return listToMap(feeds, Feed::getId);
    }


    public Map<Long, Set<Long>> getAdGroupCountriesByAdGroupId(Long clientRegionId,
                                                               Collection<AdGroupForBannerOperation> adGroups) {
        GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(clientRegionId);
        return StreamEx.of(adGroups)
                .mapToEntry(AdGroupForBannerOperation::getId, AdGroupForBannerOperation::getGeo)
                .nonNullValues()
                .mapValues(geoTree::getModerationCountries)
                .toMap();
    }

    public Map<Long, Set<Long>> getDuplicatedCreativeIdsByAdGroupId(int shard,
                                                                    List<PerformanceBanner> changedBanners) {

        Set<Long> adGroupIds = listToSet(changedBanners, PerformanceBanner::getAdGroupId);
        List<PerformanceBanner> oldBanners = getPerformanceBannersByGroupIds(shard, adGroupIds);

        Set<Long> changedBannerIds = StreamEx.of(changedBanners).map(PerformanceBanner::getId).nonNull().toSet();
        List<PerformanceBanner> unchangedBanners = StreamEx.of(oldBanners)
                .filter(banner -> !changedBannerIds.contains(banner.getId()))
                .toList();

        // полный набор баннеров (т.е. то, что должно получится после выполнения валидируемой операции)
        List<PerformanceBanner> newBanners = StreamEx.of(changedBanners).append(unchangedBanners).toList();

        Map<Long, List<Long>> creativeIdsByAdGroupId = StreamEx.of(newBanners)
                .filter(banner -> (banner.getAdGroupId() != null) && (banner.getCreativeId() != null))
                .mapToEntry(PerformanceBanner::getAdGroupId, PerformanceBanner::getCreativeId)
                .grouping();

        return EntryStream.of(creativeIdsByAdGroupId)
                .mapValues(creativeIds -> listToSet(findDuplicates(creativeIds), Function.identity()))
                .toMap();
    }

    private List<PerformanceBanner> getPerformanceBannersByGroupIds(int shard, Set<Long> adGroupIds) {
        List<Banner> banners = bannerRepository.getBannersByGroupIds(shard, adGroupIds);
        return filterModelsOfClass(banners, PerformanceBanner.class);
    }

    public Map<Long, CpmPriceCampaign> getCpmPriceCampaigns(int shard, Collection<CommonCampaign> campaigns) {
        Set<Long> cpmPriceCampaignIds = campaigns.stream()
                .filter(campaign -> campaign.getType() == CampaignType.CPM_PRICE)
                .map(CommonCampaign::getId)
                .collect(toSet());

        return (Map<Long, CpmPriceCampaign>) campaignTypedRepository.getTypedCampaignsMap(shard, cpmPriceCampaignIds);
    }

    public Map<Long, PricePackage> getPricePackages(int shard, Collection<CommonCampaign> campaigns) {
        Set<Long> getPricePackageIds = getCpmPriceCampaigns(shard, campaigns).values().stream()
                .map(CpmPriceCampaign::getPricePackageId)
                .collect(toSet());
        return pricePackageRepository.getPricePackages(getPricePackageIds);
    }

}
