package ru.yandex.direct.core.entity.banner.type.creative;

import java.util.HashSet;
import java.util.Map;
import java.util.function.Function;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.BannerWithCreative;
import ru.yandex.direct.core.entity.banner.model.CpcVideoBanner;
import ru.yandex.direct.core.entity.banner.model.CpmAudioBanner;
import ru.yandex.direct.core.entity.banner.model.CpmBanner;
import ru.yandex.direct.core.entity.banner.model.CpmGeoPinBanner;
import ru.yandex.direct.core.entity.banner.model.CpmIndoorBanner;
import ru.yandex.direct.core.entity.banner.model.CpmOutdoorBanner;
import ru.yandex.direct.core.entity.banner.model.ImageBanner;
import ru.yandex.direct.core.entity.banner.model.MobileAppBanner;
import ru.yandex.direct.core.entity.banner.model.PerformanceBanner;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelChangesValidationBuilder;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.banner.model.BannerWithCreative.CREATIVE_ID;
import static ru.yandex.direct.core.entity.banner.model.BannerWithCreative.SHOW_TITLE_AND_BODY;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.cpcVideoNotFound;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.creativeNotFound;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.requiredCreativesWithVideoAdditionTypeOnly;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.videoExtensionNotFound;
import static ru.yandex.direct.core.entity.banner.type.creative.BannerCreativeValidator.optionalCreativeNewBannerValidator;
import static ru.yandex.direct.core.entity.banner.type.creative.BannerCreativeValidator.requiredCreativeNewBannerValidator;
import static ru.yandex.direct.core.entity.banner.type.creative.BannerWithCreativeUtils.getAllowedCreativeTypesByClass;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notTrue;


@Component
public class BannerWithCreativeValidatorProvider {

    /**
     * Получить валидатор для проверки изменений креатива в баннере
     *
     * @param clientCreativesByIds    - мапа creativeId -> Creative, в которой содержатся старые и новые креативы
     * @param oldCreativeIdByBannerId - содержит только те креативы, которые были в баннерах до применения изменений
     */
    public Validator<ModelChanges<BannerWithCreative>, Defect> creativeModificationValidator(
            Function<ModelChanges<BannerWithCreative>, Class<? extends BannerWithCreative>>
                    getRuntimeClassFunction,
            Map<Long, Creative> clientCreativesByIds,
            Map<Long, Long> oldCreativeIdByBannerId) {

        return mc -> {
            Class<? extends BannerWithCreative> bannerClass = getRuntimeClassFunction.apply(mc);
            boolean isCreativeChanged = mc.isPropChanged(CREATIVE_ID);
            boolean isModificationCheckRequired =
                    bannerClass == CpmBanner.class || bannerClass == ImageBanner.class;

            if (!isCreativeChanged || !isModificationCheckRequired) {
                return ValidationResult.success(mc);
            }

            ModelChangesValidationBuilder<BannerWithCreative> vb = ModelChangesValidationBuilder.of(mc);

            vb.item(CREATIVE_ID).checkBy(
                    optionalCreativeNewBannerValidator(
                            clientCreativesByIds,
                            bannerClass,
                            null,
                            creativeNotFound()));

            vb.checkBy(new BannerCreativeModificationValidator(
                    clientCreativesByIds,
                    oldCreativeIdByBannerId,
                    bannerClass), When.isValid());

            return vb.getResult();
        };
    }

    public Validator<BannerWithCreative, Defect> creativeValidator(
            BannerWithCreativeValidationContainer container) {
        return bannerWithCreative -> {
            ModelItemValidationBuilder<BannerWithCreative> ivb = ModelItemValidationBuilder.of(bannerWithCreative);
            ivb.item(CREATIVE_ID)
                    .checkBy(creativeValidator(bannerWithCreative, container));
            ivb.item(SHOW_TITLE_AND_BODY)
                    .check(
                            notTrue(),
                            When.isFalse(container.isFeatureEnabledForClient(FeatureName.DISABLE_VIDEO_CREATIVE))
                    );
            return ivb.getResult();
        };
    }


    private Validator<Long, Defect> creativeValidator(BannerWithCreative banner,
                                                      BannerWithCreativeValidationContainer container) {

        if (banner instanceof CpmAudioBanner
                || banner instanceof CpmOutdoorBanner
                || banner instanceof CpmIndoorBanner) {
            return requiredCreativeNewBannerValidator(container.getCreativesByIds(), banner.getClass());
        }

        if (banner instanceof CpcVideoBanner) {
            return requiredCreativeNewBannerValidator(
                    container.getCreativesByIds(),
                    banner.getClass(),
                    cpcVideoNotFound(),
                    null);
        }

        if (banner instanceof ImageBanner) {
            var allowedCreativeTypes = new HashSet<>(getAllowedCreativeTypesByClass(banner.getClass(), null));

            return optionalCreativeNewBannerValidator(
                    container.getCreativesByIds(),
                    allowedCreativeTypes,
                    creativeNotFound(),
                    creativeNotFound());
        }

        if (banner instanceof TextBanner) {
            return optionalCreativeNewBannerValidator(
                    container.getCreativesByIds(),
                    TextBanner.class,
                    videoExtensionNotFound(),
                    requiredCreativesWithVideoAdditionTypeOnly()
                    );
        }

        if (banner instanceof PerformanceBanner) {
            return creativeId -> {
                ItemValidationBuilder<Long, Defect> builder = ItemValidationBuilder.of(creativeId);
                builder
                        .checkBy(requiredCreativeNewBannerValidator(container.getCreativesByIds(), banner.getClass()))
                        .checkBy(new PerformanceBannerCreativeValidator(
                                container.getCreativesByIds(),
                                container.getPerformanceFeedBusinessTypeByAdGroupId(),
                                container.getPerformanceAdGroupCountriesByAdGroupId(),
                                container.getPerformanceDuplicatedCreativeIdsByAdGroupId(),
                                container.getAdGroupId(banner)), When.isValid());
                return builder.getResult();
            };
        }

        if (banner instanceof CpmBanner) {
            return creativeId -> {
                ItemValidationBuilder<Long, Defect> builder = ItemValidationBuilder.of(creativeId);
                builder
                        .checkBy(requiredCreativeNewBannerValidator(
                                container.getCreativesByIds(),
                                banner.getClass(),
                                creativeNotFound(),
                                creativeNotFound()))
                        .checkBy(new CpmBannerCreativeValidator(
                                        container.getCreativesByIds(),
                                        container.isNonSkippableCpmVideoAdGroup(banner),
                                        container.getAdGroupType(banner),
                                        container.getCampaignType(banner),
                                        container.isCopy()),
                                When.isValid());

                if (container.getCampaignType(banner) == CampaignType.CPM_PRICE) {

                    var cpmPriceCampaign = container.getCpmPriceCampaigns().get(container.getCampaignId(banner));
                    builder
                            .checkBy(new CpmBannerCreativePriceSalesValidator(
                                            container.getCreativesByIds(),
                                            cpmPriceCampaign,
                                            container.getAdGroupType(banner),
                                            container.getPricePackages().get(cpmPriceCampaign.getPricePackageId())),
                                    When.isValid());
                }

                return builder.getResult();
            };
        }

        if (banner instanceof CpmGeoPinBanner) {
            return creativeId -> {
                ItemValidationBuilder<Long, Defect> builder = ItemValidationBuilder.of(creativeId);
                builder
                        .checkBy(requiredCreativeNewBannerValidator(
                                container.getCreativesByIds(),
                                banner.getClass(),
                                creativeNotFound(),
                                creativeNotFound()))
                        .checkBy(new CpmGeoPinBannerCreativeValidator(container.getCreativesByIds()),
                                When.isValid());
                return builder.getResult();
            };
        }

        if (banner instanceof MobileAppBanner) {
            return optionalCreativeNewBannerValidator(
                    container.getCreativesByIds(),
                    MobileAppBanner.class,
                    videoExtensionNotFound(),
                    requiredCreativesWithVideoAdditionTypeOnly());
        }

        throw new IllegalArgumentException("Unknown banner type. Add validation for this type");
    }
}
