package ru.yandex.direct.core.entity.banner.type.creative;

import java.util.Map;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.banner.type.creative.BannerWithCreativeConstraints.creativeIsNotAdminRejected;
import static ru.yandex.direct.core.entity.banner.type.creative.BannerWithCreativeConstraints.creativeMatchesAdGroupSkippability;
import static ru.yandex.direct.core.entity.banner.type.creative.BannerWithCreativeConstraints.creativeTypeIsCanvasOrHtml5;
import static ru.yandex.direct.core.entity.banner.type.creative.BannerWithCreativeConstraints.creativeTypeIsCpmAudioCreative;
import static ru.yandex.direct.core.entity.banner.type.creative.BannerWithCreativeConstraints.creativeTypeIsCpmVideoCreative;
import static ru.yandex.direct.core.entity.banner.type.creative.BannerWithCreativeConstraints.creativeTypeIsHtml5;
import static ru.yandex.direct.core.entity.banner.type.creative.CreativeFormatValidator.VALID_CPM_FORMATS;
import static ru.yandex.direct.core.entity.banner.type.creative.CreativeFormatValidator.VALID_CPM_GEOPRODUCT_FORMATS;
import static ru.yandex.direct.core.entity.banner.type.creative.CreativeFormatValidator.VALID_CPM_YNDX_FRONTPAGE_FORMATS;

public class CpmBannerCreativeValidator implements Validator<Long, Defect> {

    private final Map<Long, Creative> accessibleCreativesByIds;
    private final AdGroupType adGroupType;
    private final CampaignType campaignType;
    private final boolean isNonSkippableCpmVideoAdGroup;
    private final boolean isCopy;

    public CpmBannerCreativeValidator(Map<Long, Creative> accessibleCreativesByIds,
                                      boolean isNonSkippableCpmVideoAdGroup,
                                      AdGroupType adGroupType,
                                      CampaignType campaignType,
                                      boolean isCopy) {
        this.accessibleCreativesByIds = accessibleCreativesByIds;
        this.isNonSkippableCpmVideoAdGroup = isNonSkippableCpmVideoAdGroup;
        this.adGroupType = adGroupType;
        this.campaignType = campaignType;
        this.isCopy = isCopy;
    }

    @Override
    public ValidationResult<Long, Defect> apply(Long creativeId) {
        ItemValidationBuilder<Long, Defect> builder = ItemValidationBuilder.of(creativeId);

        builder
                .check(creativeIsNotAdminRejected(accessibleCreativesByIds), When.isValid());


        if (adGroupType == AdGroupType.CPM_BANNER && campaignType != CampaignType.CPM_PRICE) {
            builder
                    .check(creativeTypeIsCanvasOrHtml5(accessibleCreativesByIds), When.isValid())
                    .checkBy(new CreativeFormatValidator(accessibleCreativesByIds, VALID_CPM_FORMATS),
                            When.isValidAnd(When.isFalse(isCopy)));
        }

        if (adGroupType == AdGroupType.CPM_GEOPRODUCT) {
            builder
                    .check(creativeTypeIsCanvasOrHtml5(accessibleCreativesByIds), When.isValid())
                    .checkBy(new CreativeFormatValidator(accessibleCreativesByIds, VALID_CPM_GEOPRODUCT_FORMATS),
                            When.isValidAnd(When.isFalse(isCopy)));
        }

        if (adGroupType == AdGroupType.CPM_VIDEO && campaignType != CampaignType.CPM_PRICE) {
            builder
                    .check(creativeTypeIsCpmVideoCreative(accessibleCreativesByIds), When.isValid());
        }

        if (campaignType == CampaignType.CPM_BANNER && adGroupType == AdGroupType.CPM_VIDEO) {
            builder
                    .check(creativeMatchesAdGroupSkippability(accessibleCreativesByIds, isNonSkippableCpmVideoAdGroup),
                            When.isValid());
        }

        if (adGroupType == AdGroupType.CPM_AUDIO) {
            builder
                    .check(creativeTypeIsCpmAudioCreative(accessibleCreativesByIds), When.isValid());
        }

        if (adGroupType == AdGroupType.CPM_YNDX_FRONTPAGE && campaignType == CampaignType.CPM_YNDX_FRONTPAGE) {
            builder
                    .check(creativeTypeIsHtml5(accessibleCreativesByIds), When.isValid())
                    .checkBy(new CreativeFormatValidator(accessibleCreativesByIds, VALID_CPM_YNDX_FRONTPAGE_FORMATS),
                            When.isValidAnd(When.isFalse(isCopy)));
        }


        return builder.getResult();
    }
}
