package ru.yandex.direct.core.entity.banner.type.creative;

import java.util.Map;
import java.util.Set;

import com.google.common.collect.ImmutableSet;

import ru.yandex.direct.core.entity.banner.type.creative.model.CreativeSize;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.DefaultValidator;

import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.inconsistentCreativeFormat;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

public class CreativeFormatValidator implements DefaultValidator<Long> {
    public static final Set<CreativeSize> VALID_CPM_YNDX_FRONTPAGE_FORMATS =
            ImmutableSet.of(new CreativeSize(2 * 728L, 2 * 90L),
                    new CreativeSize(2 * 320L, 2 * 67L));

    public static final Set<CreativeSize> VALID_CPM_GEOPRODUCT_FORMATS =
            ImmutableSet.of(new CreativeSize(320L, 50L),
                    new CreativeSize(320L, 67L),
                    new CreativeSize(640L, 100L),
                    new CreativeSize(640L, 134L));

    public static final Set<CreativeSize> VALID_CPM_FORMATS =
            ImmutableSet
                    .<CreativeSize>builder()
                    .add(new CreativeSize(240L, 400L))
                    .add(new CreativeSize(240L, 600L))
                    .add(new CreativeSize(300L, 250L))
                    .add(new CreativeSize(300L, 500L))
                    .add(new CreativeSize(300L, 600L))
                    .add(new CreativeSize(336L, 280L))
                    .add(new CreativeSize(320L, 50L))
                    .add(new CreativeSize(320L, 100L))
                    .add(new CreativeSize(320L, 480L))
                    .add(new CreativeSize(728L, 90L))
                    .add(new CreativeSize(480L, 320L))
                    .add(new CreativeSize(970L, 250L))
                    .add(new CreativeSize(1000L, 120L))
                    .add(new CreativeSize(300L, 300L))
                    .add(new CreativeSize(160L, 600L))
                    .add(new CreativeSize(0L, 120L))
                    .add(new CreativeSize(0L, 150L))
                    .build();

    private final Map<Long, Creative> accessibleCreativesByIds;
    private final Set<CreativeSize> validFormats;

    public CreativeFormatValidator(Map<Long, Creative> accessibleCreativesByIds, Set<CreativeSize> validFormats) {
        this.accessibleCreativesByIds = accessibleCreativesByIds;
        this.validFormats = validFormats;
    }

    @Override
    public ValidationResult<Long, Defect> apply(Long creativeId) {
        return ItemValidationBuilder.of(creativeId, Defect.class)
                .check(validateCreativeFormat(accessibleCreativesByIds))
                .getResult();
    }

    /**
     * Проверяет валидность формата(ширины/высоты) креатива, должен соответствовать формату из набора валидных
     */
    private Constraint<Long, Defect> validateCreativeFormat(
            Map<Long, Creative> accessibleCreativesByIds) {
        return fromPredicate(creativeId -> {
                    Creative creative = accessibleCreativesByIds.get(creativeId);
                    return creative == null || isValidCreativeFormat(creative);
                },
                inconsistentCreativeFormat());
    }

    private Boolean isValidCreativeFormat(Creative creative) {
        return validFormats.contains(new CreativeSize(creative.getWidth(), creative.getHeight()))
                || validFormats.contains(new CreativeSize(0L, creative.getHeight()));
    }
}
