package ru.yandex.direct.core.entity.banner.type.displayhref;

import com.google.common.base.CharMatcher;

import ru.yandex.direct.core.entity.banner.model.BannerWithHref;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithDisplayHref;
import ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstants;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static com.google.common.base.CharMatcher.anyOf;
import static ru.yandex.direct.core.entity.banner.service.BannerUtils.templateCountIn;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.duplicateSpecCharsInDisplayHref;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.emptyDisplayHref;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.invalidDisplayHrefUsage;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.maxLengthDisplayHref;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.restrictedCharsInDisplayHref;
import static ru.yandex.direct.utils.TextConstants.BEL_LETTERS;
import static ru.yandex.direct.utils.TextConstants.COMMON_UKR_BEL_LETTERS;
import static ru.yandex.direct.utils.TextConstants.DE_LETTERS;
import static ru.yandex.direct.utils.TextConstants.KAZ_LETTERS;
import static ru.yandex.direct.utils.TextConstants.LAT_LETTERS;
import static ru.yandex.direct.utils.TextConstants.NUMBERS;
import static ru.yandex.direct.utils.TextConstants.RUS_LETTERS;
import static ru.yandex.direct.utils.TextConstants.TR_LETTERS;
import static ru.yandex.direct.utils.TextConstants.UKR_LETTERS;
import static ru.yandex.direct.validation.Predicates.empty;
import static ru.yandex.direct.validation.Predicates.not;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

public class BannerWithDisplayHrefConstraints {

    /**
     * Максимальная длина отображаемой ссылки без учёта вхождения {@link BannerTextConstants#TEMPLATE_MARKER_SYMBOL}
     * (самостоятельный диез учитывается)
     */
    public static final int MAX_LENGTH_DISPLAY_HREF = 20;

    /**
     * Символы, доступные в отображаемых ссылках
     */
    public static final String ALLOW_BANNER_DISPLAY_HREF_LETTERS_STR =
            LAT_LETTERS + RUS_LETTERS + UKR_LETTERS + BEL_LETTERS + COMMON_UKR_BEL_LETTERS + TR_LETTERS + KAZ_LETTERS
                    + DE_LETTERS + NUMBERS + "-№/%" + BannerTextConstants.TEMPLATE_MARKER_SYMBOL;

    private static final CharMatcher ALLOW_BANNER_DISPLAY_HREF_LETTERS_MATCHER =
            anyOf(ALLOW_BANNER_DISPLAY_HREF_LETTERS_STR);

    private BannerWithDisplayHrefConstraints() {}

    public static Constraint<String, Defect> displayHrefIsNotEmpty() {
        return fromPredicate(not(empty()), emptyDisplayHref());
    }

    /**
     * Проверка максимальной длины отображаемой ссылки без учёта вхождения шаблонных меток
     * {@link BannerTextConstants#TEMPLATE_MARKER_SYMBOL} (самостоятельный диез учитывается)
     */
    public static Constraint<String, Defect> displayHrefHasValidLength() {
        return fromPredicate(
                displayHref -> displayHref.length() <= MAX_LENGTH_DISPLAY_HREF + 2 * templateCountIn(displayHref),
                maxLengthDisplayHref(MAX_LENGTH_DISPLAY_HREF));
    }

    /**
     * Проверка на отсутствие двойного минуса ("--") и двойного слэша ("//") в отображаемой ссылке
     */
    public static Constraint<String, Defect> displayHrefNotContainsDuplicatedSpecialChars() {
        return fromPredicate(
                displayHref -> !displayHref.contains("--") && !displayHref.contains("//"),
                duplicateSpecCharsInDisplayHref());
    }

    /**
     * Проверка на наличие недопустимых символов в отображаемой ссылке
     */
    public static Constraint<String, Defect> displayHrefCharsAreAllowed() {
        return displayHref -> {
            if (displayHref == null || ALLOW_BANNER_DISPLAY_HREF_LETTERS_MATCHER.matchesAllOf(displayHref)) {
                return null;
            }

            return restrictedCharsInDisplayHref(ALLOW_BANNER_DISPLAY_HREF_LETTERS_MATCHER.removeFrom(displayHref));
        };
    }

    /**
     * Проверка наличия ссылки, если отображаемая ссылка задана
     */
    public static Constraint<String, Defect> hrefIsSetIfDisplayHrefNotNull(BannerWithHref banner) {
        return fromPredicate(displayHref -> displayHref == null || banner.getHref() != null, invalidDisplayHrefUsage());
    }

    public static Constraint<String, Defect> hrefIsSetIfDisplayHrefNotNullOld(OldBannerWithDisplayHref banner) {
        return fromPredicate(displayHref -> displayHref == null || banner.getHref() != null, invalidDisplayHrefUsage());
    }
}
