package ru.yandex.direct.core.entity.banner.type.displayhref;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannerAdditionalActionsContainer;
import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerDisplayHrefStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithDisplayHrefModeration;
import ru.yandex.direct.core.entity.moderation.service.ModerationService;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.core.entity.banner.model.BannerWithDisplayHrefModeration.DISPLAY_HREF;
import static ru.yandex.direct.core.entity.banner.model.BannerWithDisplayHrefModeration.DISPLAY_HREF_STATUS_MODERATE;
import static ru.yandex.direct.core.entity.banner.model.BannerWithDisplayHrefModeration.ID;
import static ru.yandex.direct.core.entity.banner.model.BannerWithDisplayHrefModeration.STATUS_MODERATE;
import static ru.yandex.direct.core.entity.banner.service.BannerUtils.getCampaignIdToBannerIds;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class DefaultBannerWithDisplayHrefModerationProcessor
        extends BannerWithDisplayHrefModerationProcessor<BannerWithDisplayHrefModeration> {

    private final ModerationService moderationService;

    @Autowired
    public DefaultBannerWithDisplayHrefModerationProcessor(ModerationService moderationService) {
        this.moderationService = moderationService;
    }

    @Override
    public Class<BannerWithDisplayHrefModeration> getProcessedClass() {
        return BannerWithDisplayHrefModeration.class;
    }

    @Override
    public void process(DSLContext context, BannerAdditionalActionsContainer additionalActionsContainer,
                        BannersUpdateOperationContainer container,
                        List<AppliedChanges<BannerWithDisplayHrefModeration>> appliedChanges) {
        filterList(appliedChanges, this::isDisplayHrefNeedModeration)
                .forEach(ac -> ac.modify(DISPLAY_HREF_STATUS_MODERATE, BannerDisplayHrefStatusModerate.READY));

        // удаленные displayHrefs нужно удалить из очереди отправки на модерацию
        var bannersWithDeletedDisplayHref = filterAndMapList(appliedChanges, ac -> ac.deleted(DISPLAY_HREF),
                AppliedChanges::getModel);
        var bannerIds = mapList(bannersWithDeletedDisplayHref, ID::get);
        var campaignIdToBannerIds = getCampaignIdToBannerIds(bannersWithDeletedDisplayHref, banner -> true);

        moderationService.clearDisplayHrefsModeration(context, container.getClientId(),
                bannerIds, campaignIdToBannerIds);
    }

    private boolean isDisplayHrefNeedModeration(AppliedChanges<BannerWithDisplayHrefModeration> appliedChanges) {
        if (appliedChanges.getNewValue(DISPLAY_HREF) == null) {
            return false;
        }

        // Отправляем на модерацию непустой displayHref не только когда он изменился,
        // но и когда существенно изменился баннер (и будет отправлен на модерацию).
        // Так нужно, потому что displayHref проверяют в контексте баннера.
        return appliedChanges.changed(DISPLAY_HREF) ||
                appliedChanges.getNewValue(STATUS_MODERATE) == BannerStatusModerate.READY;
    }

}

