package ru.yandex.direct.core.entity.banner.type.flags;

import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.MapDifference;
import com.google.common.collect.Maps;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.banner.model.Age;
import ru.yandex.direct.core.entity.banner.model.BabyFood;
import ru.yandex.direct.core.entity.banner.model.BannerFlags;
import ru.yandex.direct.core.entity.banner.model.old.OldBanner;
import ru.yandex.direct.core.entity.banner.model.old.OldTextBanner;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.cannotChangeBannerFlagsFromAgeToOtherTypes;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.cannotChangeBannerFlagsFromBabyFoodToOtherTypes;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.clientCannotSetBannerFlags;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.insufficientRights;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

public class BannerWithFlagsConstraints {

    public static final Set<String> ALLOWED_FLAGS = ImmutableSet.of(
            BannerFlags.AGE.getKey(), BannerFlags.BABY_FOOD.getKey());

    public static final BannerFlags EMPTY_FLAGS = new BannerFlags();

    private BannerWithFlagsConstraints() {}

    public static Constraint<BannerFlags, Defect> immutableFlagsCannotBeChanged(@Nullable BannerFlags oldFlags) {
        return fromPredicate(
                newFlags -> ALLOWED_FLAGS.containsAll(getChangedFlags(newFlags, nvl(oldFlags, EMPTY_FLAGS))),
                insufficientRights()
        );
    }

    private static Set<String> getChangedFlags(BannerFlags newFlags, BannerFlags oldFlags) {
        MapDifference<String, String> diff = Maps.difference(oldFlags.getFlags(), newFlags.getFlags());

        return StreamEx.of(diff.entriesDiffering(), diff.entriesOnlyOnLeft(), diff.entriesOnlyOnRight())
                .flatMap(entries -> entries.keySet().stream())
                .toSet();
    }

    /**
     * Проверяет, не было ли смены типа флагов баннера (метки возраста) - тип "Возрастной рейтинг" (AGE) можно менять
     * только на такой же, и тип "Ограничения по возрасту ребенка в месяцах" (BABY_FOOD) можно менять только на
     * такой же. Так как сейчас можно текстовому баннеру поставить только два типа, а удалить значение нельзя - мы
     * проверяем попытку смены с одного типа флагов на другой.
     *
     * @param oldFlags старое значение флагов
     */
    public static Constraint<BannerFlags, Defect> textBannerFlagsTypeCannotBeChanged(
            @Nullable BannerFlags oldFlags) {

        return newFlags -> {
            if (newFlags == null) {
                return null;
            }

            BabyFood babyFoodBefore = null;
            Age ageBefore = null;
            if (oldFlags != null) {
                babyFoodBefore = oldFlags.get(BannerFlags.BABY_FOOD);
                ageBefore = oldFlags.get(BannerFlags.AGE);
            }

            BabyFood babyFoodAfter = newFlags.get(BannerFlags.BABY_FOOD);
            Age ageAfter = newFlags.get(BannerFlags.AGE);

            if (ageBefore == null && babyFoodBefore == null && (ageAfter != null || babyFoodAfter != null)) {
                return clientCannotSetBannerFlags();
            }
            if (ageBefore != null && babyFoodBefore == null && babyFoodAfter != null) {
                return cannotChangeBannerFlagsFromAgeToOtherTypes();
            }
            if (babyFoodBefore != null && ageBefore == null && ageAfter != null) {
                return cannotChangeBannerFlagsFromBabyFoodToOtherTypes();
            }
            return null;
        };
    }

    public static <B extends OldBanner> Constraint<ModelChanges<B>, Defect> textBannerFlagsTypeCannotBeChangedOld(
            Map<Long, OldTextBanner> unmodifiedModels) {
        return after -> {
            if (after == null) {
                return null;
            }

            BannerFlags flagsBefore = unmodifiedModels.get(after.getId()).getFlags();

            return textBannerFlagsTypeCannotBeChanged(flagsBefore).apply(after.getChangedProp(OldBanner.FLAGS));
        };
    }
}
