package ru.yandex.direct.core.entity.banner.type.flags;

import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.BannerWithFlags;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.validation.builder.ListItemValidator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelChangesValidationBuilder;

import static ru.yandex.direct.core.entity.banner.model.BannerWithFlags.FLAGS;
import static ru.yandex.direct.core.entity.banner.type.flags.BannerWithFlagsConstraints.immutableFlagsCannotBeChanged;
import static ru.yandex.direct.core.entity.banner.type.flags.BannerWithFlagsConstraints.textBannerFlagsTypeCannotBeChanged;
import static ru.yandex.direct.rbac.RbacRole.PLACER;
import static ru.yandex.direct.rbac.RbacRole.SUPER;
import static ru.yandex.direct.rbac.RbacRole.SUPPORT;

@Component
@ParametersAreNonnullByDefault
public class BannerWithFlagsValidatorProvider {
    private static final Set<RbacRole> ROLES_ALLOWED_TO_CHANGE_BANNER_FLAGS = ImmutableSet.of(SUPER, PLACER, SUPPORT);

    public ListItemValidator<ModelChanges<BannerWithFlags>, Defect> bannerFlagsChangesValidator(
            Map<Long, BannerWithFlags> unmodifiedModels, RbacRole operatorRole) {
        return (index, modelChanges) -> {
            var mcvb = ModelChangesValidationBuilder.of(modelChanges);
            var oldFlags = unmodifiedModels.get(modelChanges.getId()).getFlags();

            if (ROLES_ALLOWED_TO_CHANGE_BANNER_FLAGS.contains(operatorRole)) {
                // Тут очень сильно ограничены возможности по редактированию флагов суперами и саппортами. В "старом"
                // интерфейсе эти роли могут много больше, это прописано в перле, но в Java это пока не реализовано.
                // В новом интерфейсе возможности менять флаги пока нет (и в обозримом будущем не планируется), а в АПИ
                // есть ограниченные возможности по изменению флагов для текстовых баннеров и для РМП.
                mcvb.item(FLAGS).check(immutableFlagsCannotBeChanged(oldFlags));
            } else {
                if (TextBanner.class.isAssignableFrom(unmodifiedModels.get(modelChanges.getId()).getClass())) {
                    // В текстовых баннерах разрешаем только менять значения у флагов AGE и BABY_FOOD. Добавлять/удалять
                    // эти, как и любые другие флаги, нельзя.
                    mcvb.item(FLAGS).check(textBannerFlagsTypeCannotBeChanged(oldFlags));
                } else {
                    // Во всех баннерах, кроме текстового, можно добавлять/удалять/изменять флаги AGE и BABY_FOOD.
                    // На момент написания этого комантария этот код используется только для РМП-баннеров, изменяемых
                    // через АПИ.
                    mcvb.item(FLAGS).check(immutableFlagsCannotBeChanged(oldFlags));
                }
            }
            return mcvb.getResult();
        };
    }
}
