package ru.yandex.direct.core.entity.banner.type.geoflag;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannersOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithOnlyGeoFlag;
import ru.yandex.direct.core.entity.banner.repository.type.AbstractBannerDefaultRepositoryTypeSupport;
import ru.yandex.direct.core.entity.banner.repository.type.ModifiedPaths;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.grut.objects.proto.client.Schema;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.banner.repository.BannerRepositoryConstants.Opts.GEO_FLAG;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;

@Component
public class BannerWithOnlyGeoFlagRepositoryTypeSupport
        extends AbstractBannerDefaultRepositoryTypeSupport<BannerWithOnlyGeoFlag> {

    private static final Set<ModelProperty<? super BannerWithOnlyGeoFlag, ?>> GRUT_PROPERTIES =
            Set.of(BannerWithOnlyGeoFlag.GEO_FLAG);

    @Autowired
    public BannerWithOnlyGeoFlagRepositoryTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider, BannerWithOnlyGeoFlag.GEO_FLAG, BANNERS.OPTS,
                opts -> opts.contains(GEO_FLAG), BannerWithOnlyGeoFlagRepositoryTypeSupport::bannerOptsToDb);
    }

    @Override
    public Class<BannerWithOnlyGeoFlag> getTypeClass() {
        return BannerWithOnlyGeoFlag.class;
    }

    /**
     * Получает значение ppc.banners.opts из заданных полей баннера
     *
     * @return значение ppc.banners.opts для записи в базу
     */
    private static String bannerOptsToDb(Boolean geoFlag) {
        List<String> opts = new ArrayList<>();
        if (geoFlag) {
            opts.add(GEO_FLAG);
        }
        return StringUtils.join(opts, ",");
    }

    @Override
    public Set<ModelProperty<? super BannerWithOnlyGeoFlag, ?>> getGrutSupportedProperties() {
        return GRUT_PROPERTIES;
    }

    @Override
    public Map<Long, ModifiedPaths> applyToGrutObjects(
            @NotNull Map<Long, Schema.TBannerV2.Builder> bannerBuilders,
            @NotNull Collection<AppliedChanges<BannerWithOnlyGeoFlag>> appliedChangesList,
            @NotNull BannersOperationContainer operationContainer) {
        Map<Long, ModifiedPaths> modifiedPathsMap = new HashMap<>();
        for (var appliedChanges : appliedChangesList) {
            if (appliedChanges.getPropertiesForUpdate().contains(BannerWithOnlyGeoFlag.GEO_FLAG)) {
                Long id = appliedChanges.getModel().getId();
                var newValue = appliedChanges.getNewValue(BannerWithOnlyGeoFlag.GEO_FLAG);
                ModifiedPaths modifiedPaths;
                if (newValue != null) {
                    Schema.TBannerV2.Builder bannerBuilder = bannerBuilders.get(id);
                    bannerBuilder.getSpecBuilder().getFlagsBuilder().setGeoflag(newValue);
                    modifiedPaths = new ModifiedPaths(Set.of("/spec/flags/geoflag"), emptySet());
                } else {
                    modifiedPaths = new ModifiedPaths(emptySet(), Set.of("/spec/flags/geoflag"));
                }
                modifiedPathsMap.put(id, modifiedPaths);
            }
        }
        return modifiedPathsMap;
    }
}
