package ru.yandex.direct.core.entity.banner.type.href;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Name;
import org.jooq.Record;
import org.jooq.Select;
import org.jooq.Table;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.domain.model.Domain;
import ru.yandex.direct.core.entity.vcard.model.Phone;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusarch;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusbssynced;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatuspostmoderate;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType;
import ru.yandex.direct.dbschema.ppc.tables.Banners;
import ru.yandex.direct.dbschema.ppc.tables.Domains;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.Collections.emptySet;
import static org.jooq.impl.DSL.coalesce;
import static org.jooq.impl.DSL.lower;
import static org.jooq.impl.DSL.name;
import static ru.yandex.direct.core.entity.vcard.repository.VcardMappings.phoneFromDb;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_DYNAMIC;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.DOMAINS;
import static ru.yandex.direct.dbschema.ppc.Tables.FILTER_DOMAIN;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.Tables.VCARDS;
import static ru.yandex.direct.jooqmapper.JooqMapperUtils.makeCaseStatement;
import static ru.yandex.direct.validation.Predicates.notNull;

@Repository
@ParametersAreNonnullByDefault
public class BannerDomainRepository {

    public static final Name DYNAMIC_MAIN_DOMAIN = name("dynamic_main_domain");

    // Когда у баннера изменился домен, нам нужно отправить баннер на модерацию.
    // Но архивные баннеры и черновики не нужно отправлять.
    private static final Condition BANNER_SHOULD_BE_MODERATED = BANNERS.STATUS_MODERATE.notEqual(BannersStatusmoderate.New)
            .and(BANNERS.STATUS_ARCH.equal(BannersStatusarch.No));

    private final DslContextProvider ppcDslContextProvider;
    private final ShardHelper shardHelper;

    @Autowired
    public BannerDomainRepository(DslContextProvider ppcDslContextProvider,
                                     ShardHelper shardHelper) {
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.shardHelper = shardHelper;
    }

    public Set<Long> getBannerIdsByDomain(int shard, String domain) {
        String reverseDomain = StringUtils.reverse(domain);
        return ppcDslContextProvider.ppc(shard)
                .select(BANNERS.BID)
                .from(BANNERS)
                .where(BANNERS.REVERSE_DOMAIN.eq(reverseDomain))
                .and(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No))
                .fetchSet(BANNERS.BID);
    }

    /**
     * Изменить отображаемый домен для всех баннеров кампании
     *
     * @param campaignId id кампании
     * @param domain     отображаемый домен
     * @return id баннеров, у которых изменился домен
     */
    public List<Long> changeCampaignBannersDomains(int shard, long campaignId, Domain domain) {
        return ppcDslContextProvider.ppc(shard).transactionResult(config -> {
            List<Long> bannerIds = getCampaignBannerIdsForDomainChange(shard, campaignId);
            config.dsl().update(BANNERS)
                    .set(BANNERS.DOMAIN_ID, domain.getId())
                    .set(BANNERS.DOMAIN, domain.getDomain())
                    .set(BANNERS.REVERSE_DOMAIN, domain.getReverseDomain())
                    .set(BANNERS.STATUS_BS_SYNCED, BannersStatusbssynced.No)
                    .set(BANNERS.STATUS_MODERATE, getBannersStatusModerateValue())
                    .set(BANNERS.STATUS_POST_MODERATE, getBannersStatusPostModerateValue())
                    .where(BANNERS.BID.in(bannerIds))
                    .execute();
            return bannerIds;
        });
    }

    /**
     * Изменить отображаемый домен для баннеров
     *
     */
    public void changeBannersDomains(int shard, Map<Long, Domain> bannerIdToHrefWithDomain) {
        Set<Long> bannerIds = bannerIdToHrefWithDomain.keySet();

        Map<Long, Long> domainIdByBannerId = mapValues(bannerIdToHrefWithDomain, Domain::getId);
        Map<Long, String> domainByBannerId = mapValues(bannerIdToHrefWithDomain, Domain::getDomain);
        Map<Long, String> reverseDomainByBannerId = mapValues(bannerIdToHrefWithDomain, Domain::getReverseDomain);

        ppcDslContextProvider.ppc(shard)
                .update(BANNERS)
                .set(BANNERS.DOMAIN_ID, makeCaseStatement(BANNERS.BID, BANNERS.DOMAIN_ID, domainIdByBannerId))
                .set(BANNERS.DOMAIN, makeCaseStatement(BANNERS.BID, BANNERS.DOMAIN, domainByBannerId))
                .set(BANNERS.REVERSE_DOMAIN,
                        makeCaseStatement(BANNERS.BID, BANNERS.REVERSE_DOMAIN, reverseDomainByBannerId))
                .where(BANNERS.BID.in(bannerIds))
                .execute();
    }

    /**
     * Изменить домен для переданных баннеров
     * href используется при выборе баннеров для обновления
     */
    public void changeBannersDomainsAfterCheckRedirect(int shard,
                                                       Map<Long, Pair<String, Domain>> bannerIdToHrefWithDomain) {
        if (bannerIdToHrefWithDomain.isEmpty()) {
            return;
        }

        // ссылка могла измениться пока проверяли редиректы
        Condition hrefCondition = EntryStream.of(bannerIdToHrefWithDomain)
                .mapKeyValue((bannerId, pair) -> BANNERS.BID.eq(bannerId)
                        .and(BANNERS.HREF.eq(pair.getLeft())))
                .reduce(Condition::or)
                .orElse(DSL.falseCondition());

        var domainByBannerId = mapValues(bannerIdToHrefWithDomain, p -> p.getRight().getDomain());
        var reverseDomainByBannerId = mapValues(bannerIdToHrefWithDomain, p -> p.getRight().getReverseDomain());
        var domainIdByBannerId = mapValues(bannerIdToHrefWithDomain, p -> p.getRight().getId());

        ppcDslContextProvider.ppc(shard)
                .update(BANNERS)
                .set(BANNERS.DOMAIN_ID, makeCaseStatement(BANNERS.BID, BANNERS.DOMAIN_ID, domainIdByBannerId))
                .set(BANNERS.DOMAIN, makeCaseStatement(BANNERS.BID, BANNERS.DOMAIN, domainByBannerId))
                .set(BANNERS.REVERSE_DOMAIN,
                        makeCaseStatement(BANNERS.BID, BANNERS.REVERSE_DOMAIN, reverseDomainByBannerId))
                .set(BANNERS.STATUS_BS_SYNCED, BannersStatusbssynced.No)
                .set(BANNERS.STATUS_MODERATE, getBannersStatusModerateValue())
                .set(BANNERS.STATUS_POST_MODERATE, getBannersStatusPostModerateValue())
                .where(BANNERS.BID.in(bannerIdToHrefWithDomain.keySet()))
                .and(hrefCondition)
                .execute();
    }

    private static Field<BannersStatusmoderate> getBannersStatusModerateValue() {
        return DSL.iif(BANNER_SHOULD_BE_MODERATED, BannersStatusmoderate.Ready, BANNERS.STATUS_MODERATE);
    }

    private Field<BannersStatuspostmoderate> getBannersStatusPostModerateValue() {
        return DSL.iif(
                BANNER_SHOULD_BE_MODERATED
                        .and(BANNERS.STATUS_POST_MODERATE.notEqual(BannersStatuspostmoderate.Rejected)),
                BannersStatuspostmoderate.No, BANNERS.STATUS_POST_MODERATE
        );
    }

    private static <T, R> Map<Long, R> mapValues(Map<Long, T> sourceMap, Function<T, R> valuesMapper) {
        return EntryStream.of(sourceMap).mapValues(valuesMapper).toMap();
    }

    private List<Long> getCampaignBannerIdsForDomainChange(int shard, long campaignId) {
        return ppcDslContextProvider.ppc(shard)
                .select(BANNERS.BID)
                .from(BANNERS)
                .where(BANNERS.CID.eq(campaignId))
                .and(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No))
                .and(BANNERS.HREF.isNotNull())
                .fetch(BANNERS.BID);
    }

    public Map<Long, String> getFilterDomainByBannerIdMap(ClientId clientId, Set<Long> bannerIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return ppcDslContextProvider.ppc(shard)
                .select(BANNERS.BID, FILTER_DOMAIN.FILTER_DOMAIN_)
                .from(BANNERS)
                .join(FILTER_DOMAIN).on(FILTER_DOMAIN.DOMAIN.eq(BANNERS.DOMAIN))
                .where(BANNERS.BID.in(bannerIds))
                .fetchMap(BANNERS.BID, FILTER_DOMAIN.FILTER_DOMAIN_);
    }

    /**
     * Получить домены для баннеров или, если для баннера домена нет, псевдодомен на основе телефона из визитки.
     *
     * @param context
     * @param bannerIds
     * @return Map{bannerId -> domain}
     */
    public Map<Long, String> getDomainsForStat(DSLContext context, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return new HashMap<>();
        }

        Map<Long, Record> domainDataByBannerId = buildDomainsDataQuery(context, bannerIds).fetchMap(BANNERS.BID);

        return EntryStream.of(domainDataByBannerId)
                .mapValues(rec -> {
                    String domain = null;
                    if (rec.get(DYNAMIC_MAIN_DOMAIN) != null) {
                        domain = (String) rec.get(DYNAMIC_MAIN_DOMAIN);
                    } else if (rec.get(FILTER_DOMAIN.FILTER_DOMAIN_) != null) {
                        domain = rec.get(FILTER_DOMAIN.FILTER_DOMAIN_);
                    } else if (rec.get(BANNERS.DOMAIN) != null) {
                        domain = rec.get(BANNERS.DOMAIN);
                    } else if (rec.get(VCARDS.PHONE) != null) {
                        Phone phone = phoneFromDb(rec.get(VCARDS.PHONE));
                        domain = phone.getCountryCode() + phone.getCityCode() + phone.getPhoneNumber() + ".phone";
                    }
                    return domain;
                })
                .filterValues(notNull())
                .toMap();
    }

    private static Select<Record> buildDomainsDataQuery(DSLContext context, Collection<Long> bannerIds) {
        Field[] fields = {
                BANNERS.BID,
                FILTER_DOMAIN.FILTER_DOMAIN_,
                BANNERS.DOMAIN,
                DOMAINS.DOMAIN.as(DYNAMIC_MAIN_DOMAIN),
                VCARDS.PHONE
        };
        Condition dynamicAdGroupJoinCondition =
                PHRASES.ADGROUP_TYPE.equal(PhrasesAdgroupType.dynamic).and(ADGROUPS_DYNAMIC.PID.eq(PHRASES.PID));
        Table<Record> table = BANNERS
                .join(PHRASES).on(PHRASES.PID.eq(BANNERS.PID))
                .leftJoin(ADGROUPS_DYNAMIC).on(dynamicAdGroupJoinCondition)
                .leftJoin(DOMAINS).on(DOMAINS.DOMAIN_ID.eq(ADGROUPS_DYNAMIC.MAIN_DOMAIN_ID))
                .leftJoin(VCARDS).on(VCARDS.VCARD_ID.eq(BANNERS.VCARD_ID))
                .leftJoin(FILTER_DOMAIN).on(FILTER_DOMAIN.DOMAIN.eq(BANNERS.DOMAIN));
        return context
                .select(fields)
                .from(table)
                .where(BANNERS.BID.in(bannerIds));
    }

    /**
     * Получить множество уникальных доменов баннеров и главных доменов динамических объявлений определенных кампаний.
     * Домены приведены к нижнему регистру.
     * <p>
     * Параметр {@code offset} не реализован, потому что основное применение метода - давать ответ на вопрос:
     * "уникален ли используемый в кампаниях домен или нет".
     *
     * @param shard       шард
     * @param campaignIds коллекция id кампаний, в которых производить поиск
     * @param limit       ограничение на количество записей
     * @return множество уникальных доменов
     */
    public Set<String> getUniqueBannersDomainsByCampaignIds(int shard, Collection<Long> campaignIds, int limit) {
        if (campaignIds.isEmpty()) {
            return emptySet();
        }

        Field<String> domain = lower(coalesce(Domains.DOMAINS.DOMAIN, Banners.BANNERS.DOMAIN));
        return ppcDslContextProvider.ppc(shard)
                .selectDistinct(domain)
                .from(Banners.BANNERS)
                .leftJoin(ADGROUPS_DYNAMIC).on(ADGROUPS_DYNAMIC.PID.eq(Banners.BANNERS.PID))
                .leftJoin(Domains.DOMAINS).on(Domains.DOMAINS.DOMAIN_ID.eq(ADGROUPS_DYNAMIC.MAIN_DOMAIN_ID))
                .where(Banners.BANNERS.CID.in(campaignIds)
                        .and(Banners.BANNERS.DOMAIN.isNotNull()
                                .or(Domains.DOMAINS.DOMAIN.isNotNull())))
                .limit(limit)
                .fetchSet(domain);
    }
}
