package ru.yandex.direct.core.entity.banner.type.href;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupForBannerOperation;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.model.BannerWithHref;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.domain.service.DomainService;
import ru.yandex.direct.core.entity.trustedredirects.service.TrustedRedirectsService;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.core.entity.banner.model.BannerWithHref.DOMAIN;
import static ru.yandex.direct.core.entity.banner.model.BannerWithHref.DOMAIN_ID;
import static ru.yandex.direct.core.entity.banner.model.BannerWithHref.HREF;
import static ru.yandex.direct.core.entity.banner.type.href.BannerWithHrefUtils.HREF_HAS_COEF_GOAL_CONTEXT_ID;
import static ru.yandex.direct.core.entity.banner.type.href.BannerWithHrefUtils.HREF_HAS_PHRASEID;
import static ru.yandex.direct.core.entity.banner.type.href.BannerWithHrefUtils.toUnicodeDomain;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.filterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class BannerHrefAndDomainProcessor {

    private final DomainService domainService;
    private final BannerCommonRepository bannerCommonRepository;
    private final TrustedRedirectsService trustedRedirectsService;
    private final AdGroupRepository adGroupRepository;
    private final BannersUrlHelper bannersUrlHelper;

    @Autowired
    public BannerHrefAndDomainProcessor(DomainService domainService,
                                           BannerCommonRepository bannerCommonRepository,
                                           TrustedRedirectsService trustedRedirectsService,
                                           AdGroupRepository adGroupRepository,
                                           BannersUrlHelper bannersUrlHelper) {
        this.domainService = domainService;
        this.bannerCommonRepository = bannerCommonRepository;
        this.trustedRedirectsService = trustedRedirectsService;
        this.adGroupRepository = adGroupRepository;
        this.bannersUrlHelper = bannersUrlHelper;
    }

    void processBannersHrefs(List<BannerWithHref> banners) {
        banners.forEach(banner -> banner.setHref(transformHrefToUnicode(banner)));
    }

    void processBannersChangesHrefs(List<AppliedChanges<BannerWithHref>> appliedChanges) {
        appliedChanges.forEach(ac -> ac.modify(HREF, transformHrefToUnicode(ac.getModel())));
    }

    private String transformHrefToUnicode(BannerWithHref banner) {
        return bannersUrlHelper.toUnicodeUrl(banner.getHref());
    }

    void processBannersDomains(List<BannerWithHref> banners, boolean isOperatorInternal) {
        banners.forEach(banner -> {
            String domain;
            if (banner.getDomain() != null && isOperatorInternal) {
                domain = toUnicodeDomain(banner.getDomain());
            } else {
                domain = bannersUrlHelper.extractHostFromHrefWithWwwOrNull(banner.getHref());
            }
            banner.setDomain(domain);
        });
    }

    void processBannersChangesDomains(List<AppliedChanges<BannerWithHref>> appliedChanges) {
        appliedChanges.forEach(ac -> {
            String domain;
            if (ac.changed(DOMAIN)) {
                domain = toUnicodeDomain(ac.getNewValue(DOMAIN));
            } else {
                domain = bannersUrlHelper.extractHostFromHrefWithWwwOrNull(ac.getNewValue(HREF));
            }
            ac.modify(DOMAIN, domain);
        });
    }

    void storeBannersDomainsAndSetDomainIds(int shard, List<BannerWithHref> banners) {
        var domainToId = storeDomainsAndGetIds(shard, banners);
        banners.forEach(banner ->
                banner.setDomainId(domainToId.get(banner.getDomain()))
        );
    }

    void storeDomainsAndSetDomainIds(int shard, List<AppliedChanges<BannerWithHref>> appliedChangesList) {
        var domainToId = storeDomainsAndGetIds(shard, mapList(appliedChangesList, AppliedChanges::getModel));
        appliedChangesList.forEach(ac ->
                ac.modify(DOMAIN_ID, domainToId.get(ac.getNewValue(DOMAIN))));
    }

    private Map<String, Long> storeDomainsAndGetIds(int shard, List<BannerWithHref> banners) {
        List<String> domains = StreamEx.of(banners)
                .map(BannerWithHref::getDomain)
                .filter(Objects::nonNull)
                .toList();
        return domainService.addDomains(shard, domains);
    }

    void updateFilterDomain(DSLContext dslContext, List<BannerWithHref> banners) {
        domainService.updateFilterDomain(dslContext, mapList(banners, DOMAIN::get));
    }

    void addBannersToRedirectCheckQueue(int shard, List<BannerWithHref> banners) {
        bannerCommonRepository.addToRedirectCheckQueue(shard, getBannerIdsWithTrustedDomains(banners));
    }

    void addBannersChangesToRedirectCheckQueue(int shard, List<AppliedChanges<BannerWithHref>> appliedChangesList) {
        bannerCommonRepository.addToRedirectCheckQueue(shard,
                getBannerIdsWithTrustedDomains(mapList(appliedChangesList, AppliedChanges::getModel))
        );
    }

    private List<Long> getBannerIdsWithTrustedDomains(List<BannerWithHref> banners) {
        return StreamEx.of(banners)
                .filter(banner -> trustedRedirectsService.isDomainTrusted(banner.getDomain()))
                .map(BannerWithHref::getId)
                .toList();
    }

    Set<Long> getAdGroupsToResendToBS(List<BannerWithHref> banners) {
        return StreamEx.of(banners)
                .filter(this::isBannerHrefContainsSpecialParameters)
                .map(BannerWithHref::getAdGroupId)
                .toSet();
    }

    private boolean isBannerHrefContainsSpecialParameters(BannerWithHref banner) {
        return HREF_HAS_COEF_GOAL_CONTEXT_ID.test(banner.getHref()) || isBannerHrefHasPhraseId(banner);
    }

    private boolean isBannerHrefHasPhraseId(BannerWithHref banner) {
        return HREF_HAS_PHRASEID.test(banner.getHref());
    }

    Set<Long> getAdGroupsToSetHasPhraseIdHref(Collection<AdGroupForBannerOperation> adGroups,
                                              List<BannerWithHref> banners) {
        var groupIdsWithoutHasPhraseIdHref = StreamEx.of(adGroups)
                .filter(adGroup -> adGroup.getHasPhraseIdHref() == null || !adGroup.getHasPhraseIdHref())
                .map(AdGroupForBannerOperation::getId)
                .toSet();

        return filterToSet(mapList(banners, BannerWithHref::getAdGroupId), groupIdsWithoutHasPhraseIdHref::contains);
    }

    void setAdGroupsHasPhraseIdHref(DSLContext dsl, List<BannerWithHref> banners,
                                    Collection<AdGroupForBannerOperation> agGroups) {
        adGroupRepository.setHasPhraseIdHrefs(dsl,
                getAdGroupsToSetHasPhraseIdHref(agGroups, filterList(banners, this::isBannerHrefHasPhraseId)));
    }
}
