package ru.yandex.direct.core.entity.banner.type.href;

import java.time.Duration;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.trustedredirects.service.TrustedRedirectsService;
import ru.yandex.direct.core.service.urlchecker.RedirectCheckResult;
import ru.yandex.direct.core.service.urlchecker.RedirectChecker;
import ru.yandex.direct.core.service.urlchecker.UrlCheckResult;
import ru.yandex.direct.core.service.urlchecker.UrlChecker;
import ru.yandex.direct.core.service.urlchecker.UrlToCheck;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.UrlUtils.urlDomainToAscii;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.validHref;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

/**
 * Сервис проверки доступности url'ов в баннерах.
 */
@Service
@ParametersAreNonnullByDefault
public class BannerUrlCheckService {
    private static final Logger logger = LoggerFactory.getLogger(BannerUrlCheckService.class);

    public static final long DEFAULT_TIMEOUT = 20_000L;
    public static final int REDIRECTS_LIMIT = 5;
    private static final int REDIRECTS_TRUSTED_LIMIT = 9;

    private final UrlChecker urlChecker;
    private final RedirectChecker redirectChecker;
    private final TrustedRedirectsService trustedRedirectsService;
    private final PpcProperty<Boolean> skipUrlCheckProperty;

    @Autowired
    public BannerUrlCheckService(
            PpcPropertiesSupport ppcPropertiesSupport,
            UrlChecker urlChecker,
            RedirectChecker redirectChecker,
            TrustedRedirectsService trustedRedirectsService) {
        this.urlChecker = urlChecker;
        this.redirectChecker = redirectChecker;
        this.trustedRedirectsService = trustedRedirectsService;
        skipUrlCheckProperty = ppcPropertiesSupport.get(PpcPropertyNames.SKIP_URL_CHECK, Duration.ofMinutes(1));
    }

    public MassResult<UrlCheckResult> checkUrls(List<String> urls) {
        List<String> sanitizedUrls = mapList(urls, this::sanitize);
        ValidationResult<List<String>, Defect> vr = ListValidationBuilder.<String, Defect>of(sanitizedUrls)
                .checkEachBy(urlValidator())
                .getResult();

        List<String> validUrls = getValidItems(vr);

        List<UrlCheckResult> results;
        boolean skipUrlCheck = skipUrlCheckProperty.getOrDefault(false);
        if (!skipUrlCheck) {
            results = mapList(validUrls, url -> urlChecker.isUrlReachable(createUrlToCheck(url)));
        } else {
            // всё равно простучим URL'ы, чтобы в логах сохранить результат
            validUrls.forEach(url -> urlChecker.isUrlReachable(createUrlToCheck(url)));
            logger.info("Ignore URL checks as property 'skip_url_check' enabled");
            results = mapList(validUrls, url -> new UrlCheckResult(true, null));
        }

        return MassResult
                .successfulMassAction(ValidationResult.mergeSuccessfulAndInvalidItems(vr, results, url -> null), vr);
    }

    /**
     * получение домена после редиректов
     * ссылка должна быть предварительно провалидирована
     */
    public RedirectCheckResult getRedirect(String href) {
        return getRedirect(href, null, false);
    }

    public RedirectCheckResult getRedirect(String href, boolean trustRedirectFromAnyDomain) {
        return getRedirect(href, null, trustRedirectFromAnyDomain);
    }

    public RedirectCheckResult getRedirect(String href, @Nullable String userAgent, boolean trustRedirectFromAnyDomain) {
        try {
            return redirectChecker.getRedirectUrl(createUrlToCheck(href, userAgent, trustRedirectFromAnyDomain));
        } catch (RuntimeException e) {
            logger.error("can't get redirect for href {}", href, e);
            return RedirectCheckResult.createFailResult();
        }
    }

    public UrlCheckResult isUrlReachable(String url) {
        UrlToCheck urlToCheck = createUrlToCheck(url);
        return urlChecker.isUrlReachable(urlToCheck);
    }

    /**
     * 1. Заменяет хитрые кавычки на обычные.
     * 2. Удаляет пробелы в начале/конце строки.
     * 3. Заменяет множественные пробелы одним.
     * 4. Удаляет пробелы между буквами и пунктуацией.
     *
     * @param url ссылка для нормализации
     * @return нормализованная ссылка
     */
    @Nullable
    private String sanitize(@Nullable String url) {
        if (url == null) {
            return null;
        }
        return url.trim()
                .replaceAll("[\\x{2010}\\x{2011}\\x{2012}\\x{2015}\\x96]", "-")
                .replaceAll("[\\x{203a}\\x{2039}\\x{2033}\\x{201e}\\x{201d}\\x{201c}\\x{201a}\\x{201b}\\x{2018}]", "\"")
                .replaceAll("[\\x{ab}\\x{bb}]", "\"")
                .replaceAll("[\\x{2032}`]", "\'")
                .replaceAll("\\s+", " ")
                .replaceAll("(?<=[\\w])\\s+(?=[.,;!:()?])", "")
                .replaceAll("(?<=[.,;!:()?])\\s+(?=[\\w])", "");
    }

    private Validator<String, Defect> urlValidator() {
        return url -> {
            ItemValidationBuilder<String, Defect> v = ItemValidationBuilder.of(url);
            v.check(notNull())
                    .check(notBlank(), When.isValid())
                    .check(validHref(), When.isValid());
            return v.getResult();
        };
    }

    private UrlToCheck createUrlToCheck(String url) {
        return createUrlToCheck(url, null, false);
    }

    private UrlToCheck createUrlToCheck(String url, @Nullable String userAgent, boolean trustRedirectFromAnyDomain) {
        return new UrlToCheck()
                .withUrl(stubHrefParameters(urlDomainToAscii(url)))
                .withRedirectsLimit(decideRedirectsLimit(url))
                .withTimeout(DEFAULT_TIMEOUT)
                .withUserAgent(userAgent)
                .withTrustRedirectFromAnyDomain(trustRedirectFromAnyDomain);
    }

    /**
     * Подставляет значения-заглушки вместо параметров в ссылке.
     *
     * @param url ссылка с параметрами
     * @return ссылка со значениями-заглушками
     */
    private String stubHrefParameters(String url) {
        return url
                .replaceAll("(?:(\\{(?:source|source_type|position_type|position|keyword|addphrases|region_name)\\}))+",
                        "test")
                .replaceAll("\\{region_id\\}", "123")
                .replaceAll("\\{device_type\\}", "desktop");
    }

    /**
     * Выбирает максимально допустимое количество редиректов для проверки ссылки.
     * Для известных сервисов шортенеров/трекеров это значение больше чем для обычных ссылок.
     *
     * @param url ссылка для которой надо определить максимально допустимое количество редиректов
     * @return максимально допустимое количество редиректов
     */
    private int decideRedirectsLimit(String url) {
        if (trustedRedirectsService.isTrusted(url)
                || trustedRedirectsService.checkTrackingHref(url) == TrustedRedirectsService.Result.TRUSTED) {
            return REDIRECTS_TRUSTED_LIMIT;
        }
        return REDIRECTS_LIMIT;
    }
}
