package ru.yandex.direct.core.entity.banner.type.href;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Objects;
import java.util.Optional;

import ru.yandex.direct.core.entity.banner.model.BannerWithHref;
import ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects;
import ru.yandex.direct.core.entity.trustedredirects.service.TrustedRedirectsService;
import ru.yandex.direct.core.entity.uac.model.TrackingSystem;
import ru.yandex.direct.core.entity.uac.service.trackingurl.ParserType;
import ru.yandex.direct.core.entity.uac.service.trackingurl.TrackingUrlParseService;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static org.apache.commons.lang3.StringUtils.containsIgnoreCase;
import static ru.yandex.direct.core.entity.banner.service.BannerUtils.templateCountIn;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.emptyHref;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.incorrectImpressionUrlTrackerId;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.incorrectImpressionUrlTrackingSystem;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.invalidHref;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.maxHrefLengthWithoutTemplateMarker;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.trackingSystemDomainNotSupported;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.trackingSystemDomainSupportsHttpsOnly;
import static ru.yandex.direct.core.entity.banner.type.href.BannerWithHrefConstants.DOMAIN_PATTERN;
import static ru.yandex.direct.core.validation.defects.RightsDefects.noRights;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.validHref;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

public class BannerWithHrefConstraints {

    private BannerWithHrefConstraints() {
        // only static methods
    }

    public static Constraint<String, Defect> hrefIsNotBlank() {
        return notBlank().overrideDefect(emptyHref());
    }

    /**
     * Проверка максимальной длины ссылки без учёта символа '#'
     */
    public static Constraint<String, Defect> hrefHasValidLength() {
        return hrefHasValidLength(BannerWithHrefConstants.MAX_LENGTH_HREF);
    }

    /**
     * Приватный констрейнт для проверки длины ссылки
     */
    private static Constraint<String, Defect> hrefHasValidLength(int maxLength) {
        return fromPredicate(x -> x.length() <= maxLength + 2 * templateCountIn(x),
                maxHrefLengthWithoutTemplateMarker(maxLength));
    }

    /**
     * Проверка формата ссылки.
     */
    public static Constraint<String, Defect> hrefIsValid() {
        return validHref().overrideDefect(invalidHref());
    }

    public static Constraint<String, Defect> trackingUrlContainsLogidMacros() {
        return url -> {
            boolean isAdjust =
                    extractDomain(url).map(u -> {
                        return BannerWithHrefConstants.DOMAIN_FOR_ADJUST.equals(u) ||
                                BannerWithHrefConstants.DOMAIN_FOR_ADJUST_IMPRESSION.equals(u);
                    }).orElse(false);
            if (isAdjust) {
                return null;
            } else {
                return trackingUrlContainsMacros(BannerWithHrefConstants.LOGID_MACROS).apply(url);
            }
        };
    }

    private static Optional<String> extractDomain(String url) {
        try {
            URL parsedUrl = new URL(url);
            return Optional.of(parsedUrl.getHost());
        } catch (MalformedURLException ignore) {
            return Optional.empty();
        }
    }

    public static Constraint<String, Defect> trackingUrlContainsMacros(String macros) {
        return fromPredicate(url -> containsIgnoreCase(url, "{" + macros + "}"),
                BannerDefects.trackingUrlDoesntContainMacros());
    }

    public static Constraint<ModelChanges<BannerWithHref>, Defect> domainIsNotChanged() {
        return fromPredicate(mc -> !mc.isPropChanged(BannerWithHref.DOMAIN), noRights());
    }

    public static Constraint<String, Defect> domainIsNotSet() {
        return fromPredicate(Objects::isNull, noRights());
    }

    /**
     * Проверка что домен отвечает паттерну
     * {@link BannerWithHrefConstants#DOMAIN_PATTERN}.
     */
    public static Constraint<String, Defect> domainMatchesAllowedPattern() {
        return fromPredicate(d -> DOMAIN_PATTERN.matcher(d).matches(), invalidValue());
    }

    /**
     * @see TrustedRedirectsService#checkTrackingHref
     */
    public static Constraint<String, Defect> validTrackingHref(TrustedRedirectsService trustedRedirectsService) {
        return href -> {
            if (href == null) {
                return null;
            }
            switch (trustedRedirectsService.checkTrackingHref(href)) {
                case TRUSTED:
                    return null;
                case NOT_TRUSTED:
                    return trackingSystemDomainNotSupported();
                case HTTPS_REQUIRED:
                    return trackingSystemDomainSupportsHttpsOnly();
                default:
                    throw new IllegalStateException("Unexpected result.");
            }
        };
    }

    /**
     * @see TrustedRedirectsService#checkImpressionUrl
     */
    public static Constraint<String, Defect> validImpressionUrl(TrustedRedirectsService trustedRedirectsService,
                                                                TrackingUrlParseService trackingUrlParseService,
                                                                String trackingUrl) {
        return href -> {
            if (href == null) {
                return null;
            }
            switch (trustedRedirectsService.checkImpressionUrl(href)) {
                case TRUSTED:
                    break;
                case NOT_TRUSTED:
                    return trackingSystemDomainNotSupported();
                case HTTPS_REQUIRED:
                    return trackingSystemDomainSupportsHttpsOnly();
                default:
                    throw new IllegalStateException("Unexpected result.");
            }
            if (trackingUrl == null) {
                return null;
            }
            if (!Objects.equals(trackingUrlParseService.getTrackingSystem(trackingUrl, ParserType.TRACKING_URL),
                    trackingUrlParseService.getTrackingSystem(href, ParserType.IMPRESSION_URL))) {
                return incorrectImpressionUrlTrackingSystem();
            }
            if (!Objects.equals(trackingUrlParseService.getTrackerId(trackingUrl, ParserType.TRACKING_URL),
                    trackingUrlParseService.getTrackerId(href, ParserType.IMPRESSION_URL))) {
                return incorrectImpressionUrlTrackerId();
            }
            return null;
        };
    }

    public static Constraint<String, Defect> validTrackingSystem(TrackingUrlParseService trackingUrlParseService) {
        return href -> {
            if (href == null) {
                return null;
            }

            TrackingSystem trackingSystem = trackingUrlParseService.getTrackingSystem(href, ParserType.TRACKING_URL);
            if (trackingSystem == null) {
                return trackingSystemDomainNotSupported();
            }
            return null;
        };
    }
}
