package ru.yandex.direct.core.entity.banner.type.href;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jooq.Field;
import org.jooq.Record;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannersOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithHref;
import ru.yandex.direct.core.entity.banner.repository.type.AbstractBannerRepositoryTypeSupport;
import ru.yandex.direct.core.entity.banner.repository.type.ModifiedPaths;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelperAggregator;
import ru.yandex.direct.jooqmapperhelper.UpdateHelperAggregator;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.grut.objects.proto.client.Schema;

import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;

@Component
@ParametersAreNonnullByDefault
public class BannerWithHrefRepositoryTypeSupport extends AbstractBannerRepositoryTypeSupport<BannerWithHref> {
    private static final JooqMapper<BannerWithHref> MAPPER = createMapper();

    @Autowired
    public BannerWithHrefRepositoryTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider);
    }

    @Override
    public Collection<Field<?>> getFields() {
        return MAPPER.getFieldsToRead();
    }

    @Override
    public <M extends BannerWithHref> void fillFromRecord(M model, Record record) {
        MAPPER.fromDb(record, model);
    }

    @Override
    public Class<BannerWithHref> getTypeClass() {
        return BannerWithHref.class;
    }

    @Override
    public void pushToInsert(InsertHelperAggregator insertHelperAggregator, BannerWithHref model) {
        insertHelperAggregator.getOrCreate(BANNERS).add(MAPPER, model);
    }

    @Override
    public void processUpdate(UpdateHelperAggregator updateHelperAggregator,
                              Collection<AppliedChanges<BannerWithHref>> appliedChanges) {
        updateHelperAggregator.getOrCreate(BANNERS.BID).processUpdateAll(MAPPER, appliedChanges);
    }

    private static JooqMapper<BannerWithHref> createMapper() {
        return JooqMapperBuilder.<BannerWithHref>builder()
                .readProperty(BannerWithHref.ID, fromField(BANNERS.BID))
                .map(property(BannerWithHref.HREF, BANNERS.HREF))
                .map(property(BannerWithHref.DOMAIN, BANNERS.DOMAIN))
                .map(property(BannerWithHref.DOMAIN_ID, BANNERS.DOMAIN_ID))
                .writeField(BANNERS.REVERSE_DOMAIN, fromProperty(BannerWithHref.DOMAIN).by(StringUtils::reverse))
                .build();
    }

    @Override
    public Set<ModelProperty<? super BannerWithHref, ?>> getGrutSupportedProperties() {
        return Set.of(BannerWithHref.HREF, BannerWithHref.DOMAIN);
    }

    @Override
    public Map<Long, ModifiedPaths> applyToGrutObjects(
            @NotNull Map<Long, Schema.TBannerV2.Builder> bannerBuilders,
            @NotNull Collection<AppliedChanges<BannerWithHref>> appliedChangesList,
            @NotNull BannersOperationContainer operationContainer) {
        Map<Long, ModifiedPaths> modifiedPathsMap = new HashMap<>();
        for (var appliedChanges : appliedChangesList) {
            Long id = appliedChanges.getModel().getId();
            Set<String> setPaths = new HashSet<>();
            Set<String> removePaths = new HashSet<>();
            if (appliedChanges.getPropertiesForUpdate().contains(BannerWithHref.HREF)) {
                var newValue = appliedChanges.getNewValue(BannerWithHref.HREF);
                if (newValue != null) {
                    Schema.TBannerV2.Builder bannerBuilder = bannerBuilders.get(id);
                    bannerBuilder.getSpecBuilder().setHref(newValue);
                    setPaths.add("/spec/href");
                } else {
                    removePaths.add("/spec/href");
                }
            }
            if (appliedChanges.getPropertiesForUpdate().contains(BannerWithHref.DOMAIN)) {
                var newValue = appliedChanges.getNewValue(BannerWithHref.DOMAIN);
                if (newValue != null) {
                    Schema.TBannerV2.Builder bannerBuilder = bannerBuilders.get(id);
                    bannerBuilder.getSpecBuilder().setDomain(newValue);
                    setPaths.add("/spec/domain");
                } else {
                    removePaths.add("/spec/domain");
                }
            }
            if (!setPaths.isEmpty() || !removePaths.isEmpty()) {
                modifiedPathsMap.put(id, new ModifiedPaths(setPaths, removePaths));
            }
        }
        return modifiedPathsMap;
    }
}
