package ru.yandex.direct.core.entity.banner.type.href;

import java.util.List;
import java.util.function.Predicate;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannerAdditionalActionsContainer;
import ru.yandex.direct.core.entity.banner.container.BannersModerationContainer;
import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithHref;
import ru.yandex.direct.core.entity.banner.service.type.update.AbstractBannerUpdateOperationTypeSupport;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.core.entity.banner.model.BannerWithHref.DOMAIN;
import static ru.yandex.direct.core.entity.banner.model.BannerWithHref.HREF;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

@Component
public class BannerWithHrefUpdateOperationTypeSupport
        extends AbstractBannerUpdateOperationTypeSupport<BannerWithHref> {

    private final BannerHrefAndDomainProcessor processor;

    @Autowired
    public BannerWithHrefUpdateOperationTypeSupport(BannerHrefAndDomainProcessor processor) {
        this.processor = processor;
    }

    @Override
    public Class<BannerWithHref> getTypeClass() {
        return BannerWithHref.class;
    }

    @Override
    public void beforeExecution(BannersUpdateOperationContainer container,
                                List<AppliedChanges<BannerWithHref>> appliedChanges) {
        processor.processBannersChangesHrefs(filterList(appliedChanges, ac -> ac.changedAndNotDeleted(HREF)));

        processor.processBannersChangesDomains(filterList(appliedChanges, ac ->
                ac.changed(DOMAIN) || ac.changed(HREF)));

        processor.storeDomainsAndSetDomainIds(container.getShard(),
                filterList(appliedChanges, ac -> ac.changed(DOMAIN)));
    }

    @Override
    public void addToAdditionalActionsContainer(BannerAdditionalActionsContainer additionalActionsContainer,
                                                BannersUpdateOperationContainer container,
                                                List<AppliedChanges<BannerWithHref>> appliedChanges) {
        additionalActionsContainer.addAdGroupsIdsForBSResync(processor.getAdGroupsToResendToBS(
                getBannersThatMeetCondition(appliedChanges, ac -> ac.changedAndNotDeleted(HREF))
        ));
    }

    @Override
    public void updateRelatedEntitiesInTransaction(DSLContext dslContext, BannersUpdateOperationContainer container,
                                                   List<AppliedChanges<BannerWithHref>> appliedChanges) {
        processor.updateFilterDomain(dslContext,
                getBannersThatMeetCondition(appliedChanges, ac -> ac.changedAndNotDeleted(DOMAIN)));

        processor.setAdGroupsHasPhraseIdHref(dslContext,
                getBannersThatMeetCondition(appliedChanges, ac -> ac.changedAndNotDeleted(HREF)),
                container.getAdGroups());
    }

    @Override
    public void afterExecution(BannersUpdateOperationContainer container,
                               List<AppliedChanges<BannerWithHref>> appliedChanges) {
        processor.addBannersToRedirectCheckQueue(container.getShard(),
                getBannersThatMeetCondition(appliedChanges, ac -> ac.changedAndNotDeleted(DOMAIN)));
    }

    private List<BannerWithHref> getBannersThatMeetCondition(
            List<AppliedChanges<BannerWithHref>> appliedChanges,
            Predicate<AppliedChanges<BannerWithHref>> condition) {
        return filterAndMapList(appliedChanges, condition, AppliedChanges::getModel);
    }

    @Override
    public boolean needModeration(BannersModerationContainer container,
                                  AppliedChanges<BannerWithHref> appliedChanges) {
        return appliedChanges.changed(HREF) || appliedChanges.changed(DOMAIN);
    }

    @Override
    public boolean needBsResync(AppliedChanges<BannerWithHref> appliedChanges) {
        return appliedChanges.changed(HREF) || appliedChanges.changed(DOMAIN);
    }
}
