package ru.yandex.direct.core.entity.banner.type.href;

import java.util.Optional;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupForBannerOperation;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.container.BannersOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithHref;
import ru.yandex.direct.core.entity.banner.model.ContentPromotionBanner;
import ru.yandex.direct.core.entity.banner.model.CpcVideoBanner;
import ru.yandex.direct.core.entity.banner.model.CpmAudioBanner;
import ru.yandex.direct.core.entity.banner.model.CpmBanner;
import ru.yandex.direct.core.entity.banner.model.CpmIndoorBanner;
import ru.yandex.direct.core.entity.banner.model.CpmOutdoorBanner;
import ru.yandex.direct.core.entity.banner.model.ImageBanner;
import ru.yandex.direct.core.entity.banner.model.McBanner;
import ru.yandex.direct.core.entity.banner.model.MobileAppBanner;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppTrackerTrackingSystem;
import ru.yandex.direct.core.entity.trustedredirects.service.TrustedRedirectsService;
import ru.yandex.direct.core.entity.uac.service.trackingurl.TrackingUrlParseService;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.invalidHref;
import static ru.yandex.direct.core.entity.banner.type.href.BannerWithHrefUtils.FIELD_TRACKING_URL;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_AVG_CPI;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_CPI_GOAL_ID;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.validation.builder.When.isTrue;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Component
public class BannerWithHrefValidatorProvider {

    private final TrustedRedirectsService trustedRedirectsService;
    private final TrackingUrlParseService trackingUrlParseService;

    @Autowired
    public BannerWithHrefValidatorProvider(TrustedRedirectsService trustedRedirectsService, TrackingUrlParseService trackingUrlParseService) {
        this.trustedRedirectsService = trustedRedirectsService;
        this.trackingUrlParseService = trackingUrlParseService;
    }

    public Validator<BannerWithHref, Defect> bannerWithHrefValidator(BannersOperationContainer container) {
        return bannerWithHref -> {
            boolean isMobileAppBanner = isMobileAppBanner(bannerWithHref, container);

            ModelItemValidationBuilder<BannerWithHref> ivb = ModelItemValidationBuilder.of(bannerWithHref);
            // Костыль для РМП баннеров. В api передается поле trackingUrl, а в остальных приложениях - href
            String hrefFieldName = container.isFromApi() && isMobileAppBanner
                    ? FIELD_TRACKING_URL : BannerWithHref.HREF.name();
            ivb.item(bannerWithHref.getHref(), hrefFieldName)
                    .checkBy(hrefValidator(bannerWithHref, container));
            return ivb.getResult();
        };
    }

    private Validator<String, Defect> hrefValidator(BannerWithHref banner, BannersOperationContainer container) {

        if (banner instanceof TextBanner) {

            return href -> {
                ItemValidationBuilder<String, Defect> builder = ItemValidationBuilder.of(href);
                builder
                        .checkBy(new BannerHrefValidator());
                return builder.getResult();
            };
        }

        if (banner instanceof CpcVideoBanner) {
            boolean isMobileAppBanner = isMobileAppBanner(banner, container);

            return href -> {
                ItemValidationBuilder<String, Defect> builder = ItemValidationBuilder.of(href);
                builder
                        .check(notNull(), When.isTrue(isTrackingHrefRequiredForMobileContentAd(banner, container)
                                && isMobileAppBanner))
                        .checkBy(new BannerHrefValidator())
                        .checkBy(new BannerTrackingHrefValidator(trustedRedirectsService),
                                When.isValidAnd(isTrue(isMobileAppBanner)));
                return builder.getResult();
            };
        }

        if (banner instanceof MobileAppBanner) {
            return href -> {
                ItemValidationBuilder<String, Defect> builder = ItemValidationBuilder.of(href);
                builder
                        .check(notNull(), When.isTrue(isTrackingHrefRequiredForMobileContentAd(banner, container)))
                        .checkBy(new BannerHrefValidator())
                        .checkBy(new BannerTrackingHrefValidator(trustedRedirectsService), When.isValid())
                        .checkBy(new BannerTrackingHrefMacrosValidator(),
                                When.isValidAnd(When.isTrue(container.isValidateTrackingHrefMacros())))
                        .checkBy(new BannerHrefTrackingSystemValidator(trackingUrlParseService),
                                When.isValidAnd(When.isTrue(isTrackingSystemRequiredForMobileContentAd(banner, container))));
                return builder.getResult();
            };
        }

        if (banner instanceof McBanner || banner instanceof CpmAudioBanner) {
            return href -> {
                ItemValidationBuilder<String, Defect> builder = ItemValidationBuilder.of(href);
                builder
                        .check(notNull())
                        .checkBy(new BannerHrefValidator());
                return builder.getResult();
            };
        }

        if (banner instanceof ImageBanner) {
            boolean isMobileAppBanner = isMobileAppBanner(banner, container);
            var hrefBlankDefect = isMobileAppBanner ? invalidHref() : null;
            return href -> {
                ItemValidationBuilder<String, Defect> builder = ItemValidationBuilder.of(href);
                builder
                        .check(notNull(), When.isTrue(isTrackingHrefRequiredForMobileContentAd(banner, container)
                                && isMobileAppBanner))
                        .checkBy(new BannerHrefValidator(hrefBlankDefect))
                        .checkBy(new BannerTrackingHrefValidator(trustedRedirectsService),
                                When.isValidAnd(isTrue(isMobileAppBanner)))
                        .checkBy(new BannerHrefTrackingSystemValidator(trackingUrlParseService),
                                When.isValidAnd(When.isTrue(isMobileAppBanner && isTrackingSystemRequiredForMobileContentAd(banner, container))));
                return builder.getResult();
            };
        }

        if (banner instanceof CpmBanner) {
            AdGroupType adGroupType =
                    ifNotNull(container.getAdGroup(banner), AdGroupForBannerOperation::getType);
            CampaignType campaignType = ifNotNull(container.getCampaign(banner), CommonCampaign::getType);
            boolean isCpmGeoProduct = AdGroupType.CPM_GEOPRODUCT == adGroupType;

            return href -> {
                ItemValidationBuilder<String, Defect> builder = ItemValidationBuilder.of(href);
                builder
                        .check(isNull(), When.isTrue(isCpmGeoProduct))
                        .check(notNull(), When.isTrue(campaignType == CampaignType.CPM_PRICE))
                        .checkBy(new BannerHrefValidator());
                return builder.getResult();
            };
        }

        if (banner instanceof CpmIndoorBanner || banner instanceof CpmOutdoorBanner) {
            return href -> {
                ItemValidationBuilder<String, Defect> builder = ItemValidationBuilder.of(href);
                builder
                        .check(notNull())
                        .checkBy(new BannerHrefValidator())
                        .checkBy(new BannerAdvertisingSourceValidator(), When.isValid());
                return builder.getResult();
            };
        }

        // href сами выставляем после всех валидаций в prepare
        if (banner instanceof ContentPromotionBanner) {
            return ValidationResult::success;
        }

        return new BannerHrefValidator();
    }

    private boolean isMobileAppBanner(BannerWithHref banner, BannersOperationContainer container){
        AdGroupType adGroupType = ifNotNull(container.getAdGroup(banner), AdGroupForBannerOperation::getType);
        return AdGroupType.MOBILE_CONTENT == adGroupType;
    }

    // Для РМП баннеров трекинговая ссылка бывает обязательной
    // https://st.yandex-team.ru/DIRECT-131345#5faea9e3f5ebf657682829a9
    private boolean isTrackingHrefRequiredForMobileContentAd(BannerWithHref banner,
                                                             BannersOperationContainer container) {
        CampaignWithStrategy campaignWithStrategy = container.getCampaignWithStrategy(banner);
        return Optional.ofNullable(campaignWithStrategy)
                .map(CampaignWithStrategy::getStrategy)
                .map(this::isTrackingHrefRequiredForMobileContentAd)
                .orElse(false);
    }

    private boolean isTrackingSystemRequiredForMobileContentAd(BannerWithHref banner,
                                                             BannersOperationContainer container) {
        if (!container.isFeatureEnabledForClient(FeatureName.RMP_CPI_UNDER_KNOWN_TRACKING_SYSTEM_ONLY)) {
            return false;
        }

        boolean isUniversalCampaignBanner = container.isUniversalCampaignBanner(banner);
        if (isUniversalCampaignBanner) {
            return false;
        }

        return Optional.ofNullable(container.getMobileAppTracker(banner))
                .map(t -> t.getTrackingSystem() != MobileAppTrackerTrackingSystem.OTHER)
                .orElse(false);
    }

    private boolean isTrackingHrefRequiredForMobileContentAd(DbStrategy strategy) {
        if (strategy.isSearchStop()) {
            return false;
        }
        StrategyName strategyName = strategy.getStrategyName();
        Long goalId = strategy.getStrategyData().getGoalId();
        return AUTOBUDGET_AVG_CPI.equals(strategyName)
                || (AUTOBUDGET.equals(strategyName) && DEFAULT_CPI_GOAL_ID.equals(goalId));
    }
}
