package ru.yandex.direct.core.entity.banner.type.href;

import java.net.IDN;
import java.net.MalformedURLException;
import java.net.URL;
import java.time.Duration;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.common.enums.YandexDomain;

import static java.util.Map.entry;

@Component
@ParametersAreNonnullByDefault
public class BannersUrlHelper {
    private static final String MARKET_SERVICE_NAME = "market";
    private static final String POKUPKI_MARKET_SERVICE_NAME = "pokupki.market";

    public static final List<String> YANDEX_DOMAINS = StreamEx.of(YandexDomain.values())
            .map(YandexDomain::getYandexDomain).toList();
    public static final String YANDEX_URLS_REGEX = String.format("^(?:https://|http://)?(?:www.|m.)?(%s)/([^/?#]+).*",
            String.join("|", YANDEX_DOMAINS));
    /**
     * Конечный вид этой регулярки примерно такой (доменов больше, они берутся из {@link YandexDomain}).
     * <p>
     * ^(?:https://|http://)?(?:www.|m.)?(yandex.ru|yandex.kz)/([^/?#]+).*
     * <p>
     * Регулярка имеет две группы, первая это домен, вторая - первый элемент пути.
     */
    public static final Pattern YANDEX_URLS_PATTERN = Pattern.compile(YANDEX_URLS_REGEX);

    private static final String YANDEX_DOMAINS_REGEX = String.format(
            "(?:m\\.)?((.*)\\.)?(%s)",
            String.join("|", YANDEX_DOMAINS)
    );
    public static final Pattern YANDEX_DOMAINS_PATTERN = Pattern.compile(YANDEX_DOMAINS_REGEX);

    // Для ссылок вида https://yandex.ru/<service-name>/... мы иногда хотим использовать в баннере не домен yandex.ru,
    // а специальный сервисный домен. Например, для https://yandex.ru/maps/ будет использован домен maps.yandex.ru.
    // В некоторых случаях к одному сервису относятся разные виды сервисных ссылок.
    // Например, ссылки вида https://yandex.ru/web-maps/ будут относиться к сервису maps, как и ссылки вида
    // https://yandex.ru/maps/
    public static final Map<String, String> YANDEX_URL_TO_SERVICE_MAPPINGS = Map.ofEntries(
            entry("bus", "bus"),
            entry("collections", "collections"),
            entry("efir", "efir"),
            entry("local", "local"),
            entry("maps", "maps"),
            entry("web-maps", "maps"),
            entry("pogoda", "pogoda"),
            entry("portal", "portal"),
            entry("profile", "sprav"),
            entry("sprav", "sprav"),
            entry("tutor", "tutor"),
            entry("uslugi", "uslugi"),
            entry(MARKET_SERVICE_NAME, MARKET_SERVICE_NAME)
    );

    private static final Logger logger = LoggerFactory.getLogger(BannersUrlHelper.class);

    private final PpcProperty<Map<String, String>> yandexUrlToServiceMappings;

    @Autowired
    public BannersUrlHelper(
            PpcPropertiesSupport ppcPropertiesSupport) {
        this.yandexUrlToServiceMappings = ppcPropertiesSupport.get(PpcPropertyNames.YANDEX_URL_TO_SERVICE_MAPPINGS,
                Duration.ofMinutes(10));
    }

    @Nullable
    public String extractHostFromHrefWithoutWwwOrNull(@Nullable String href) {
        return cutWww(extractHostFromHrefWithWwwOrNull(href));
    }

    @Nullable
    private String cutWww(@Nullable String domain) {
        if (domain == null) {
            return null;
        }

        if (domain.startsWith("www.")) {
            domain = domain.substring(4);
        }

        return domain;
    }

    /**
     * Формирует из ссылки домен.
     * <p>
     * Сначала проверяет регуляркой {@link BannersUrlHelper#YANDEX_URLS_REGEX} ведет ли ссылка на внутренний сервис
     * яндекса через путь вроде yandex.ru/service, и если ведет - проверяет по
     * набору {@link BannersUrlHelper#getYandexService} нужно ли приводить его к виду service.yandex.ru.
     * <p>
     * Если ссылка на внутренний сервис не ведет либо мы не знаем нужно ли ее заменять для внутренних сервисов,
     * просто возвращаем хост из текущей ссылки.
     */
    @Nullable
    public String extractHostFromHrefWithWwwOrNull(@Nullable String href) {
        if (href == null) {
            return null;
        }
        try {
            if (!href.startsWith("http://") && !href.startsWith("https://")) {
                // добавим протокол, если его нет. Следующие операции работают с полноценными URL
                href = "https://" + href;
            }
            // Если нужно, делаем подмену домена для яндексовых сервисов
            Matcher matcher = YANDEX_URLS_PATTERN.matcher(href);
            if (matcher.matches()) {
                String path = matcher.group(2);
                if (getYandexService(path) != null) {
                    return getYandexService(path) + "." + matcher.group(1);
                }
            }
            String domain = new URL(href).getHost();
            return StringUtils.trimToNull(domain);
        } catch (MalformedURLException e) {
            // something wrong with current href, skip it
            logInvalidUrlError(href, e);
            return null;
        }
    }

    public boolean isYandexMarketDomain(@Nullable String domain) {
        String yandexService = extractYandexServiceFromDomain(domain);
        return MARKET_SERVICE_NAME.equals(yandexService) || POKUPKI_MARKET_SERVICE_NAME.equals(yandexService);
    }

    @Nullable
    public String extractYandexServiceFromDomain(@Nullable String domain) {
        if (domain == null) {
            return null;
        }
        Matcher matcher = YANDEX_DOMAINS_PATTERN.matcher(domain);
        if (matcher.matches()) {
            return matcher.group(2);
        }
        return null;
    }

    /**
     * Получить название сервиса, которому соответствует путь после yandex.ru, например, yandex.ru/maps
     * Значение можно переопределить используя ppcproperty
     * если path есть в базе, то будет использоваться он, иначе — значение из кода
     *
     * @param path значение пути, следующим за yandex.tld/
     */
    @Nullable
    private String getYandexService(String path) {
        var override = yandexUrlToServiceMappings.getOrDefault(Collections.emptyMap());
        return override.getOrDefault(path, YANDEX_URL_TO_SERVICE_MAPPINGS.getOrDefault(path, null));
    }

    /**
     * Приведение к юникод-отображению ссылки, если невозможно, возвращается переданный параметр
     */
    public String toUnicodeUrl(String href) {
        String host = extractHostFromHrefWithWwwOrNull(href);
        return host == null ? href : href.replaceFirst(host, IDN.toUnicode(host));
    }

    private static void logInvalidUrlError(String href, Exception ex) {
        logger.error("URI build or parse error for value {}. Href must be already validated. Skip error.", href, ex);
    }
}
