package ru.yandex.direct.core.entity.banner.type.image;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.BannerImageOpts;
import ru.yandex.direct.core.entity.banner.model.NewStatusImageModerate;
import ru.yandex.direct.core.entity.banner.model.StatusBannerImageModerate;
import ru.yandex.direct.core.entity.banner.type.bannerimage.BannerImageConverterKt;
import ru.yandex.direct.core.entity.banner.type.image.container.ImageBannerBsData;
import ru.yandex.direct.dbschema.ppc.Tables;
import ru.yandex.direct.dbschema.ppc.enums.BannerImagesStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.dbschema.ppc.tables.BannerImages.BANNER_IMAGES;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.Images.IMAGES;

@Repository
@ParametersAreNonnullByDefault
public class BannerImageRepository {

    private final DslContextProvider dslContextProvider;

    private static final int CHUNK_SIZE = 1000;

    private static final Set<BannersBannerType> CAN_BE_SENT_TO_BS_IMAGE_BANNER_TYPES = Set.of(
            BannersBannerType.text,
            BannersBannerType.dynamic,
            BannersBannerType.mobile_content);

    @Autowired
    public BannerImageRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    public Map<Long, List<Long>> getImageIdsByBannerIds(int shard, Collection<Long> bannerIds) {
        return getImageIdsByBannerIds(dslContextProvider.ppc(shard), bannerIds);
    }

    public Map<Long, List<Long>> getImageIdsByBannerIds(DSLContext context, Collection<Long> bannerIds) {
        return context
                .select(IMAGES.BID, IMAGES.IMAGE_ID).from(IMAGES)
                .where(IMAGES.BID.in(bannerIds))
                .fetchGroups(IMAGES.BID, IMAGES.IMAGE_ID);
    }

    /**
     * отправка изображений на модерацию
     */
    public void setImagesStatusModerate(DSLContext context, Collection<Long> imageIds, NewStatusImageModerate status) {
        if (imageIds.isEmpty()) {
            return;
        }

        context
                .update(IMAGES)
                .set(IMAGES.STATUS_MODERATE, NewStatusImageModerate.toSource(status))
                .where(IMAGES.IMAGE_ID.in(imageIds))
                .execute();
    }

    /**
     * Установка статуса модерации картинки
     *
     * @param bannerIds список id баннеров
     */
    public void setBannerImagesStatusModerate(DSLContext context, Collection<Long> bannerIds,
                                              StatusBannerImageModerate statusModerate) {
        if (bannerIds.isEmpty()) {
            return;
        }
        context
                .update(Tables.BANNER_IMAGES)
                .set(Tables.BANNER_IMAGES.STATUS_MODERATE, StatusBannerImageModerate.toSource(statusModerate))
                .where(Tables.BANNER_IMAGES.BID.in(bannerIds))
                .execute();
    }

    /**
     * Возвращает список image_id у элементов с DATE_ADDED <= {@param borderDateAdded} и STATUS_SHOW = No и
     * BANNER_ID = 0
     *
     * @param shard           шард
     * @param borderDateAdded граничное время
     */
    public List<Long> getBannerImagesIdsForDelete(int shard, LocalDateTime borderDateAdded) {
        return dslContextProvider.ppc(shard)
                .select(BANNER_IMAGES.IMAGE_ID)
                .from(BANNER_IMAGES)
                .where(BANNER_IMAGES.DATE_ADDED.lessOrEqual(borderDateAdded))
                .and(BANNER_IMAGES.STATUS_SHOW.eq(BannerImagesStatusshow.No))
                .and(BANNER_IMAGES.BANNER_ID.eq(0L))
                .fetch(BANNER_IMAGES.IMAGE_ID);
    }

    /**
     * Очищает таблицу banner_images по IMAGE_ID {@param imageIds}
     *
     * @param shard    шард
     * @param imageIds коллекция image_id
     * @return количество удаленных строк
     */
    public int deleteBannerImagesByIds(int shard, Collection<Long> imageIds) {
        return dslContextProvider.ppc(shard)
                .deleteFrom(BANNER_IMAGES)
                .where(BANNER_IMAGES.IMAGE_ID.in(imageIds))
                .execute();
    }

    public Set<Long> getBidsByImageHash(int shard, Collection<String> imageHashes, Long clientId) {
        if (imageHashes.isEmpty()) {
            return Set.of();
        }

        return dslContextProvider.ppc(shard)
                .select(BANNER_IMAGES.BID)
                .from(BANNER_IMAGES)
                .join(BANNERS).on(BANNERS.BID.eq(BANNER_IMAGES.BID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .where(BANNER_IMAGES.IMAGE_HASH.in(imageHashes))
                .and(CAMPAIGNS.CLIENT_ID.eq(clientId))
                .fetchSet(BANNER_IMAGES.BID);
    }

    /**
     * Получить imageId (отсылаемый в БК "bid" картиночной копии баннера) и BannerID по указанным bids.
     * Возвращает значения только для тех типов баннеров, у которых есть отсылаемая картиночная копия.
     *
     * @return map bid -> Pair.of(imageId, BannerId)
     */
    public Map<Long, ImageBannerBsData> getBannerImageIdsFromBids(int shard, Collection<Long> bids) {
        if (bids.isEmpty()) {
            return Map.of();
        }

        Map<Long, ImageBannerBsData> result = new HashMap<>();
        for (var chunkedBids : Iterables.partition(bids, CHUNK_SIZE)) {
            result.putAll(
                    dslContextProvider.ppc(shard)
                            .select(BANNER_IMAGES.BID, BANNER_IMAGES.IMAGE_ID, BANNER_IMAGES.BANNER_ID,
                                    BANNERS.BANNER_ID, BANNER_IMAGES.OPTS)
                            .from(BANNER_IMAGES)
                            .join(BANNERS).on(BANNERS.BID.eq(BANNER_IMAGES.BID))
                            .where(BANNER_IMAGES.BID.in(chunkedBids))
                            .and(BANNERS.BANNER_TYPE.in(CAN_BE_SENT_TO_BS_IMAGE_BANNER_TYPES))
                            .fetchMap(BANNER_IMAGES.BID, record ->
                                    new ImageBannerBsData(
                                            record.get(BANNER_IMAGES.BID),
                                            record.get(BANNER_IMAGES.IMAGE_ID),
                                            record.get(BANNERS.BANNER_ID),
                                            record.get(BANNER_IMAGES.BANNER_ID),
                                            hasSingleAdToBs(BannerImageConverterKt.optsFromDb(record.get(BANNER_IMAGES.OPTS))))
                            )
            );
        }
        return result;
    }

    private boolean hasSingleAdToBs(@Nullable Set<BannerImageOpts> bannerImageOpts) {
        return bannerImageOpts != null && bannerImageOpts.contains(BannerImageOpts.SINGLE_AD_TO_BS);
    }

    /**
     * Получить BannerID изображений по указанным bids
     *
     * @return map BannerID -> bid
     */
    public Map<Long, Long> getBannerIdsFromBids(int shard, Collection<Long> bids) {
        return getBannerIdsFromBids(dslContextProvider.ppc(shard), bids);
    }

    public Map<Long, Long> getBannerIdsFromBids(DSLContext context, Collection<Long> bids) {
        Map<Long, Long> result = new HashMap<>();
        for (var chunkedBids : Iterables.partition(bids, CHUNK_SIZE)) {
            result.putAll(
                    context
                            .select(BANNER_IMAGES.BANNER_ID, BANNER_IMAGES.BID)
                            .from(BANNER_IMAGES)
                            .where(BANNER_IMAGES.BID.in(chunkedBids))
                            .and(BANNER_IMAGES.BANNER_ID.ne(0L))
                            .fetchMap(BANNER_IMAGES.BANNER_ID, BANNER_IMAGES.BID)
            );
        }
        return result;
    }

    public Set<Long> getBannerIdsWithImages(int shard,
                                            Collection<Long> bannerIds) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        Set<Long> result = new HashSet<>();
        for (var chunkedBids : Iterables.partition(bannerIds, CHUNK_SIZE)) {
            result.addAll(dslContext
                    .select(BANNER_IMAGES.BID)
                    .from(BANNER_IMAGES)
                    .where(BANNER_IMAGES.BID.in(chunkedBids))
                    .fetch(BANNER_IMAGES.BID)
            );
        }
        return result;
    }

    public Map<Long, ClientId> getClientIdsByImageIds(int shard, Collection<Long> imageIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNER_IMAGES.IMAGE_ID, CAMPAIGNS.CLIENT_ID)
                .from(BANNER_IMAGES)
                .join(BANNERS).on(BANNERS.BID.eq(BANNER_IMAGES.BID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .where(BANNER_IMAGES.IMAGE_ID.in(imageIds))
                .and(CAMPAIGNS.CLIENT_ID.isNotNull())
                .fetchMap(BANNER_IMAGES.IMAGE_ID, r -> ClientId.fromLong(r.get(CAMPAIGNS.CLIENT_ID)));
    }

    public Map<Long, Long> getCampaignIdsByImageIdsForShard(int shard, List<Long> imageIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNER_IMAGES.IMAGE_ID, BANNERS.CID)
                .from(BANNER_IMAGES)
                .join(BANNERS).on(BANNER_IMAGES.BID.eq(BANNERS.BID))
                .where(BANNER_IMAGES.IMAGE_ID.in(imageIds))
                .fetchMap(BANNER_IMAGES.IMAGE_ID, BANNERS.CID);
    }

}
