package ru.yandex.direct.core.entity.banner.type.image;

import java.util.Map;
import java.util.Objects;

import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupForBannerOperation;
import ru.yandex.direct.core.entity.banner.model.BannerWithCreative;
import ru.yandex.direct.core.entity.banner.model.BannerWithImage;
import ru.yandex.direct.core.entity.banner.model.BannerWithImageAndCreative;
import ru.yandex.direct.core.entity.banner.model.ImageSize;
import ru.yandex.direct.core.entity.banner.model.ImageType;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.cannotChangeCreativeToImageId;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.cannotChangeImageToCreativeId;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.imageSizeModification;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.inconsistentBannerType;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.inconsistentStateBannerTypeAndAdgroupType;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.requiredButEmptyImageHashOrCreativeId;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.requiredOnlyImageOrCreativeId;
import static ru.yandex.direct.core.entity.banner.type.image.BannerWithImageValidatorProvider.ALLOWED_IMAGE_TYPES;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

public class BannerWithImageConstraints {

    private BannerWithImageConstraints() {
    }

    static Constraint<ModelChanges<BannerWithImage>, Defect> isMobileImageNotChanged(
            Map<Long, BannerWithImage> unmodifiedModels) {
        return fromPredicate(mc -> {
            var newIsMobileImage = mc.getPropIfChanged(BannerWithImage.IS_MOBILE_IMAGE);
            if (newIsMobileImage == null) {
                return true;
            }

            var oldIsMobileImage = ifNotNull(unmodifiedModels.get(mc.getId()), BannerWithImage::getIsMobileImage);
            return oldIsMobileImage == null || Objects.equals(oldIsMobileImage, newIsMobileImage);
        }, inconsistentBannerType());
    }

    static Constraint<BannerWithImage, Defect> isNewImageBannerSubtypeValid(AdGroupForBannerOperation adGroup) {
        return fromPredicate(b -> {
            if (b == null || b.getIsMobileImage() == null || adGroup == null) {
                return true;
            }
            switch (adGroup.getType()) {
                case BASE:
                    return !b.getIsMobileImage();
                case MOBILE_CONTENT:
                    return b.getIsMobileImage();
                default:
                    //другие типы групп валидируются отдельно
                    return true;
            }
        }, inconsistentStateBannerTypeAndAdgroupType());
    }

    static Constraint<ModelChanges<BannerWithImage>, Defect> imageSizeIsNotChanged(
            Map<Long, BannerWithImage> unmodifiedModels,
            Map<String, Pair<ImageType, ImageSize>> hashToTypeWithSize) {
        return fromPredicate(mc -> {
            var newHash = mc.getPropIfChanged(BannerWithImage.IMAGE_HASH);
            if (newHash == null) {
                return true;
            }

            String oldHash = unmodifiedModels.get(mc.getId()).getImageHash();
            var oldTypeWithSize = hashToTypeWithSize.get(oldHash);
            var newTypeWithSize = hashToTypeWithSize.get(newHash);

            if (newTypeWithSize == null || oldTypeWithSize == null
                    || !ALLOWED_IMAGE_TYPES.contains(newTypeWithSize.getLeft())) {
                //проверяем размеры только разрешенных изображений
                return true;
            }

            ImageSize oldSize = oldTypeWithSize.getRight();
            ImageSize newSize = newTypeWithSize.getRight();
            return oldSize == null || newSize == null || Objects.equals(oldSize, newSize);
        }, imageSizeModification());
    }

    static Constraint<BannerWithImageAndCreative, Defect> imageOrCreativeIsSet() {
        return fromPredicate(
                b -> b.getImageHash() != null || b.getCreativeId() != null,
                requiredButEmptyImageHashOrCreativeId());
    }

    static Constraint<BannerWithImageAndCreative, Defect> onlyImageOrCreativeIdIsSet() {
        return fromPredicate(
                b -> (b.getImageHash() == null && b.getCreativeId() != null)
                        || (b.getImageHash() != null && b.getCreativeId() == null),
                requiredOnlyImageOrCreativeId());
    }

    /**
     * Проверяет, не было ли смены картинки на креатив
     *
     * @param unmodifiedModels словарь баннеров до применения апдейта
     */
    static Constraint<ModelChanges<BannerWithImageAndCreative>, Defect> imageNotChangedToCreative(
            Map<Long, BannerWithImageAndCreative> unmodifiedModels) {

        return fromPredicate(after ->
        {
            var imageBefore = ifNotNull(unmodifiedModels.get(after.getId()), BannerWithImage::getImageHash);
            var creativeIdAfter = after.getPropIfChanged(BannerWithImageAndCreative.CREATIVE_ID);

            return !(imageBefore != null && creativeIdAfter != null);
        }, cannotChangeImageToCreativeId());
    }

    /**
     * Проверяет, не было ли смены креатива на картинку
     *
     * @param unmodifiedModels словарь баннеров до применения апдейта
     */
    static Constraint<ModelChanges<BannerWithImageAndCreative>, Defect> creativeNotChangedToImage(
            Map<Long, BannerWithImageAndCreative> unmodifiedModels) {

        return fromPredicate(after ->
        {
            var creativeIdBefore = ifNotNull(unmodifiedModels.get(after.getId()), BannerWithCreative::getCreativeId);
            var imageAfter = after.getPropIfChanged(BannerWithImageAndCreative.IMAGE_HASH);

            return !(creativeIdBefore != null && imageAfter != null);
        }, cannotChangeCreativeToImageId());
    }
}
