package ru.yandex.direct.core.entity.banner.type.image;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.JoinType;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithImage;
import ru.yandex.direct.core.entity.banner.model.NewStatusImageModerate;
import ru.yandex.direct.core.entity.banner.repository.type.AbstractFlatRelatedEntityUpsertRepositoryTypeSupport;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerImagesRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.ImagesRecord;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.JoinQuery;

import static ru.yandex.direct.core.entity.banner.type.image.BannerWithImageUtils.isImageChanged;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES;
import static ru.yandex.direct.dbschema.ppc.Tables.IMAGES;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

@Component
@ParametersAreNonnullByDefault
public class BannerWithImageRepositoryTypeSupport
        extends AbstractFlatRelatedEntityUpsertRepositoryTypeSupport
        <BannerWithImage, BannerImagesRecord> {

    private final ShardHelper shardHelper;

    @Autowired
    public BannerWithImageRepositoryTypeSupport(DslContextProvider dslContextProvider,
                                                ShardHelper shardHelper) {
        super(dslContextProvider, BANNER_IMAGES.BID, createMapper());
        this.shardHelper = shardHelper;
    }

    @Override
    public List<JoinQuery> joinQuery() {
        return List.of(
                new JoinQuery(IMAGES, JoinType.LEFT_OUTER_JOIN, IMAGES.BID.eq(BANNERS.BID)),
                new JoinQuery(PHRASES, JoinType.JOIN, PHRASES.PID.eq(BANNERS.PID))
        );
    }

    @Override
    protected void deleteEntities(DSLContext context, BannerRepositoryContainer updateParameters,
                                  Collection<AppliedChanges<BannerWithImage>> appliedChanges) {
        //удаление не предусмотрено
    }

    @Override
    public void upsertEntity(DSLContext context, Collection<BannerWithImage> banners) {
        generateAndSetNewIds(banners);

        InsertHelper<ImagesRecord> insertHelper = new InsertHelper<>(context, IMAGES)
                .addAll(getJooqMapper(), banners);

        insertHelper.onDuplicateKeyUpdate()
                .set(IMAGES.STATUS_MODERATE, MySQLDSL.values(IMAGES.STATUS_MODERATE))
                .set(IMAGES.IMAGE_HASH, MySQLDSL.values(IMAGES.IMAGE_HASH));

        insertHelper.executeIfRecordsAdded();
    }

    private void generateAndSetNewIds(Collection<BannerWithImage> bannersWithImage) {
        List<BannerWithImage> bannersWithImageWithNoId =
                filterList(bannersWithImage, b -> b.getImageId() == null);
        Iterator<Long> bannerImageIds = shardHelper.generateImageIds(bannersWithImageWithNoId.size()).iterator();

        bannersWithImageWithNoId.forEach(banner -> banner.setImageId(bannerImageIds.next()));
    }

    @Override
    protected boolean isAddEntity(BannerWithImage model) {
        return model.getImageHash() != null;
    }

    @Override
    protected boolean isDeleteEntity(AppliedChanges<BannerWithImage> appliedChange) {
        return false;
    }

    @Override
    protected boolean isUpsertEntity(AppliedChanges<BannerWithImage> ac) {
        return isImageChanged(ac);
    }

    @Override
    public Class<BannerWithImage> getTypeClass() {
        return BannerWithImage.class;
    }


    private static JooqMapper<BannerWithImage> createMapper() {
        return JooqMapperBuilder.<BannerWithImage>builder()
                .writeField(IMAGES.BID, fromProperty(BannerWithImage.ID))
                .writeField(IMAGES.PID, fromProperty(BannerWithImage.AD_GROUP_ID))
                .writeField(IMAGES.CID, fromProperty(BannerWithImage.CAMPAIGN_ID))
                .readProperty(BannerWithImage.IS_MOBILE_IMAGE, fromField(PHRASES.ADGROUP_TYPE)
                        .by(type -> type != PhrasesAdgroupType.base))
                .map(property(BannerWithImage.IMAGE_ID, IMAGES.IMAGE_ID))
                .map(property(BannerWithImage.IMAGE_HASH, IMAGES.IMAGE_HASH))
                .map(convertibleProperty(BannerWithImage.IMAGE_STATUS_MODERATE, IMAGES.STATUS_MODERATE,
                        NewStatusImageModerate::fromSource, NewStatusImageModerate::toSource))
                .build();
    }
}
