package ru.yandex.direct.core.entity.banner.type.image;

import java.util.Map;
import java.util.Set;

import one.util.streamex.EntryStream;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannersOperationContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithImage;
import ru.yandex.direct.core.entity.banner.model.ImageSize;
import ru.yandex.direct.core.entity.banner.model.ImageType;
import ru.yandex.direct.core.entity.banner.model.McBanner;
import ru.yandex.direct.core.entity.banner.service.validation.BannerImageHashValidator;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelChangesValidationBuilder;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.banner.type.image.BannerWithImageConstraints.isNewImageBannerSubtypeValid;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Component
public class BannerWithImageValidatorProvider {

    public static final Set<ImageType> ALLOWED_IMAGE_TYPES = Set.of(ImageType.IMAGE_AD);
    public static final Set<ImageSize> MC_BANNER_ALLOWED_IMAGE_SIZES = Set.of(new ImageSize()
            .withWidth(240)
            .withHeight(400));

    public Validator<ModelChanges<BannerWithImage>, Defect> imageModificationValidator(
            Map<String, Pair<ImageType, ImageSize>> imageHashToTypeWithSize,
            Map<Long, BannerWithImage> unmodifiedModels) {
        return mc -> {

            ModelChangesValidationBuilder<BannerWithImage> vb = ModelChangesValidationBuilder.of(mc);

            vb.checkBy(new NewImageModificationValidator(imageHashToTypeWithSize, unmodifiedModels));
            return vb.getResult();
        };
    }


    public Validator<BannerWithImage, Defect> imageValidator(
            BannersOperationContainer container) {
        Map<String, Pair<ImageType, ImageSize>> existingTypesWithSizes = container.getExistingImageTypesWithSizes();
        Map<String, ImageType> existingTypes = buildImageHashToImageTypeMap(existingTypesWithSizes);
        Map<String, ImageSize> existingSizes = buildImageHashToImageSizeMap(existingTypesWithSizes);

        return banner -> {
            ModelItemValidationBuilder<BannerWithImage> vb = ModelItemValidationBuilder.of(banner);
            var adGroup = container.getAdGroup(banner);

            vb.item(BannerWithImage.IMAGE_HASH)
                    .checkBy(new BannerImageHashValidator(existingTypes, ALLOWED_IMAGE_TYPES));
            vb.check(isNewImageBannerSubtypeValid(adGroup));

            if (banner instanceof McBanner) {
                vb.item(BannerWithImage.IMAGE_HASH)
                        .check(notNull())
                        .checkBy(new NewImageSizeValidator(existingSizes, MC_BANNER_ALLOWED_IMAGE_SIZES));
            }

            return vb.getResult();
        };
    }

    private Map<String, ImageType> buildImageHashToImageTypeMap(
            Map<String, Pair<ImageType, ImageSize>> imagesTypeAndSize) {
        return EntryStream.of(imagesTypeAndSize)
                .mapValues(Pair::getLeft)
                .toMap();
    }

    private Map<String, ImageSize> buildImageHashToImageSizeMap(
            Map<String, Pair<ImageType, ImageSize>> imagesTypeAndSize) {
        return EntryStream.of(imagesTypeAndSize)
                .mapValues(Pair::getRight)
                .toMap();
    }
}

