package ru.yandex.direct.core.entity.banner.type.image.moderarion;

import java.util.ArrayList;
import java.util.List;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannerAdditionalActionsContainer;
import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainer;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithImageModeration;
import ru.yandex.direct.core.entity.banner.model.NewStatusImageModerate;
import ru.yandex.direct.core.entity.moderation.service.ModerationService;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.core.entity.banner.model.BannerWithImageModeration.IMAGE_HASH;
import static ru.yandex.direct.core.entity.banner.model.BannerWithImageModeration.IMAGE_STATUS_MODERATE;
import static ru.yandex.direct.core.entity.banner.model.BannerWithImageModeration.STATUS_MODERATE;
import static ru.yandex.direct.core.entity.banner.service.BannerModerationUtils.bannerBecameDraft;
import static ru.yandex.direct.core.entity.banner.service.BannerUtils.getCampaignIdToBannerIds;
import static ru.yandex.direct.core.entity.banner.type.image.BannerWithImageUtils.isImageChanged;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class DefaultBannerWithImageModerationProcessor implements BannerWithImageModerationProcessor {

    private final ModerationService moderationService;

    @Autowired
    public DefaultBannerWithImageModerationProcessor(ModerationService moderationService) {
        this.moderationService = moderationService;
    }

    @Override
    public Class<BannerWithImageModeration> getProcessedClass() {
        return BannerWithImageModeration.class;
    }

    @Override
    public void process(
            DSLContext dsl,
            BannerAdditionalActionsContainer additionalActionsContainer, BannersUpdateOperationContainer container,
            List<AppliedChanges<BannerWithImageModeration>> appliedChanges) {

        List<BannerWithImageModeration> toDeleteFromModeration = new ArrayList<>();

        for (AppliedChanges<BannerWithImageModeration> bannerChanges : appliedChanges) {
            if (!bannerChanges.hasActuallyChangedProps() && container.getModerationMode().isDefault()) {
                continue;
            }
            var oldImageHash = bannerChanges.getOldValue(IMAGE_HASH);
            var newImageHash = bannerChanges.getNewValue(IMAGE_HASH);
            boolean imageChanged = isImageChanged(bannerChanges);

            if (newImageHash != null) {
                // Баннер является черновиком
                if (bannerBecameDraft(bannerChanges)) {
                    bannerChanges.modify(IMAGE_STATUS_MODERATE, NewStatusImageModerate.NEW);

                    if (oldImageHash != null
                            && bannerChanges.getOldValue(STATUS_MODERATE) != BannerStatusModerate.NEW) {
                        // Если картинка не была черновиком, то нужно удалить её из модерации
                        toDeleteFromModeration.add(bannerChanges.getModel());
                    }
                    continue;
                }

                if (imageChanged || bannerChanges.getNewValue(STATUS_MODERATE) == BannerStatusModerate.READY) {
                    bannerChanges.modify(IMAGE_STATUS_MODERATE, NewStatusImageModerate.READY);
                }
            } else if (imageChanged) {
                toDeleteFromModeration.add(bannerChanges.getModel());
            }

        }

        deleteFromModeration(dsl, container, toDeleteFromModeration);
    }

    private void deleteFromModeration(DSLContext dsl,
                                      BannersUpdateOperationContainer container,
                                      List<BannerWithImageModeration> toDeleteFromModeration) {

        var bannerIdsToModeration = mapList(toDeleteFromModeration, Banner::getId);
        var campaignIdToBannerIdsWithDeletedImages = getCampaignIdToBannerIds(toDeleteFromModeration,
                banner -> banner.getImageHash() == null);
        moderationService.clearImagesModeration(dsl, container.getClientId(),
                bannerIdsToModeration, campaignIdToBannerIdsWithDeletedImages);
    }
}
